/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <Foundation/Error/ErrorTypes.h>

/// Maximum length of a configuration name string, including null terminator.
#define OpenIPC_MAX_CONFIGURATION_NAME_LENGTH               (64)

/// Maximum length of a configuration parameter name string, including null terminator.
#define OpenIPC_MAX_CONFIGURATION_PARAMETER_NAME_LENGTH     (64)

/// Maximum length of a configuration parameter value string, including null terminator.
#define OpenIPC_MAX_CONFIGURATION_PARAMETER_VALUE_LENGTH    (256)

///
/// @brief An element in the active configuration.
///
/// @note Can safely be casted to DataElement.
typedef void OpenIPC_ConfigurationElement;

///
/// @brief Selects the given configuration file to be used when initializing.
///
/// @param[in]  configurationFilePath
///     The path of the config file to use for initialization.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Already_Initialized
///     The configuration was already finalized and cannot be further changed.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a configurationFilePath parameter is NULL.
/// @retval OpenIPC_Error_Cannot_Open_File
///     The file does not exist, or is inaccessible.
/// @retval OpenIPC_Error_Invalid_Configuration
///     Invalid configuration content was encountered.
typedef OpenIPC_Error (* OpenIPC_ConfigurationSelectByFile_Function)(
    IN const char* configurationFilePath);

///
/// @brief Selects the given configuration by name to be used when
///        initializing.
///
/// @param[in]  configurationName
///     The name of the configuration to use for initialization.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Already_Initialized
///     The configuration was already finalized and cannot be further changed.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a configurationName parameter is NULL.
/// @retval OpenIPC_Error_Unknown_Configuration
///     No configuration of the specified name could be found.
/// @retval OpenIPC_Error_Invalid_Configuration
///     Invalid configuration content was encountered.
typedef OpenIPC_Error (* OpenIPC_ConfigurationSelectByName_Function)(
    IN const char* configurationName);

///
/// @brief Loads the probe configuration of the specified name.
///
/// @param[in]  configurationName
///     The name of the probe configuration.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Already_Initialized
///     The configuration was already finalized and cannot be further changed.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a configurationName parameter is NULL.
/// @retval OpenIPC_Error_Unknown_Configuration
///     No configuration of the specified name could be found.
/// @retval OpenIPC_Error_Invalid_Configuration
///     Invalid configuration content was encountered.
typedef OpenIPC_Error (* OpenIPC_ConfigurationLoadProbeConfiguration_Function)(
    IN const char* configurationName);

///
/// @brief Sets whether automatic configuration is enabled.
///
/// @param[in] enabled
///     True if automatic configuration is enabled, false otherwise.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Invalid_Configuration
///     Invalid configuration content was encountered.
typedef OpenIPC_Error (* OpenIPC_ConfigurationSetAutomaticConfigurationEnabled_Function)(
    IN OpenIPC_Bool enabled);

///
/// @brief Returns whether automatic configuration is enabled.
///
/// @param[out] enabled
///     True if automatic configuration is enabled, false otherwise.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a enabled parameter is NULL.
/// @retval OpenIPC_Error_Invalid_Configuration
///     Invalid configuration content was encountered.
typedef OpenIPC_Error (* OpenIPC_ConfigurationIsAutomaticConfigurationEnabled_Function)(
    OUT OpenIPC_Bool* enabled);

///
/// @brief Sets the parameters for the selected configuration.
///
/// @param[in] parameterCount
///             The number of parameters in @a parameterNames and @a parameterValues.
/// @param[in] parameterNames
///             The names of the parameters to set.
/// @param[in] parameterValues
///             The values of the parameters to set.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Already_Initialized
///     The configuration was already finalized and cannot be further changed.
/// @retval OpenIPC_Error_Invalid_Configuration
///     One or more of the specified parameter names or values are not valid
///     for the selected configuration.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a parameterNames or @a parameterValues parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_ConfigurationSetParameterValues_Function)(
    IN uint32_t parameterCount,
    IN const char (*parameterNames)[OpenIPC_MAX_CONFIGURATION_PARAMETER_NAME_LENGTH],
    IN const char (*parameterValues)[OpenIPC_MAX_CONFIGURATION_PARAMETER_VALUE_LENGTH]);

///
/// @brief Finalizes the selected configuration with the specified parameters.
///
/// @note Once the configuration is successfully finalized, the following
///       configuration query functions can be used:
///         - GetPrimaryElement()
///         - GetPlatformControlElement()
///         - GetDeviceControlElement()
///         - GetEnDebugElement()
///         - GetProbeElement()
///         - GetProbePluginElement()
///         - GetRunControlElement()
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Already_Initialized
///     The configuration was already finalized.
typedef OpenIPC_Error (* OpenIPC_ConfigurationFinalize_Function)(
    void);

///
/// @brief Resets the configuration, allowing it to be selected, modified,
///        and finalized.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
typedef OpenIPC_Error (* OpenIPC_ConfigurationReset_Function)(
    void);

///
/// @brief Gets the name of the selected configuration.
///
/// @param[out] configurationName
///             Returns the name of the selected configuration.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a configurationName parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetName_Function)(
    OUT char configurationName[OpenIPC_MAX_CONFIGURATION_NAME_LENGTH]);

///
/// @brief Gets the primary element of the selected configuration.
///
/// @param[out] element
///             Returns a pointer to the primary element of the selected
///             configuration. The pointer will remain valid until the next
///             call to Reset().
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a element parameter is NULL.
/// @retval OpenIPC_Error_Not_Configured
///     The configuration has not been finalized.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetPrimaryElement_Function)(
    OUT const OpenIPC_ConfigurationElement** element);

///
/// @brief Gets the probe element of the selected configuration.
///
/// @param[out] element
///             Returns a pointer to the probe element of the selected
///             configuration. The pointer will remain valid until the next
///             call to Reset().
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a element parameter is NULL.
/// @retval OpenIPC_Error_Not_Configured
///     The configuration has not been finalized.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetProbeElement_Function)(
    OUT const OpenIPC_ConfigurationElement** element);

///
/// @brief Gets the probe element of the selected configuration for the
///        specified probe plug-in.
///
/// @param[out] element
///             Returns a pointer to the probe element of the selected
///             configuration. The pointer will remain valid until the next
///             call to Reset().
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a element parameter is NULL.
/// @retval OpenIPC_Error_Not_Configured
///     The configuration has not been finalized.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetProbePluginElement_Function)(
    IN const char*                           pluginName,
    OUT const OpenIPC_ConfigurationElement** element);

///
/// @brief Gets the run control element of the selected configuration.
///
/// @param[out] element
///             Returns a pointer to the run control element of the selected
///             configuration. The pointer will remain valid until the next
///             call to Reset().
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a element parameter is NULL.
/// @retval OpenIPC_Error_Not_Configured
///     The configuration has not been finalized.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetRunControlElement_Function)(
    OUT const OpenIPC_ConfigurationElement** element);

///
/// @brief Gets the number of enabled probe plug-ins in the selected
///        configuration.
///
/// @param[out] pluginCount
///             Returns the number of probe plug-ins.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a pluginCount parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetEnabledProbePluginCount_Function)(
    OUT uint32_t* pluginCount);

///
/// @brief Gets the names of enabled probe plug-ins in the selected
///        configuration.
///
/// @param[in] pluginCount
///             The number of probe plug-in names to populate in @a pluginNames.
/// @param[out] pluginNames
///             Returns the names of the probe plug-ins.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a pluginNames parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetEnabledProbePluginNames_Function)(
    IN uint32_t pluginCount,
    OUT char (*pluginNames)[OpenIPC_MAX_IDENTIFIER_LENGTH]);

///
/// @brief Gets the number of enabled run control plug-ins in the selected
///        configuration.
///
/// @param[out] pluginCount
///             Returns the number of run control plug-ins.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a pluginCount parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetEnabledRunControlPluginCount_Function)(
    OUT uint32_t* pluginCount);

///
/// @brief Gets the names of enabled run control plug-ins in the selected
///        configuration.
///
/// @param[in] pluginCount
///             The number of run control plug-in names to populate in
///             @a pluginNames.
/// @param[out] pluginNames
///             Returns the names of the run control plug-ins.
///
/// @return (OpenIPC_Error): A code indicating success or failure.
/// @retval OpenIPC_Error_No_Error
///     Indicates success.
/// @retval OpenIPC_Error_Null_Pointer
///     The @a pluginNames parameter is NULL.
typedef OpenIPC_Error (* OpenIPC_ConfigurationGetEnabledRunControlPluginNames_Function)(
    IN uint32_t pluginCount,
    OUT char (*pluginNames)[OpenIPC_MAX_IDENTIFIER_LENGTH]);
