/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
///
/// @brief A C++ wrapper around the BitData C API.
///
/// This class is defined entirely in a header file so the class itself does
/// not need to be exported from the DLL.  Since it is a wrapper to the
/// various C style functions in the DLL, this overhead is minimized.
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include <string>
#include <vector>
#include <algorithm>
#include <assert.h>

#include "BitData.h"
#include <Foundation/IndexList/OpenIPC_IndexList.h>

///
/// @brief Bit data of arbitrary length.
class OpenIPC_BitData
{
public:

    ///
    /// @brief Constructs an empty bit data value
    OpenIPC_BitData() :
        _bitData(BitData_CreateManaged(0))
    {
    }

    ///
    /// @brief Constructs a bit data value from a bit data C struct.
    ///
    /// @param bitData
    ///     The bit data to use (will be owned).
    explicit OpenIPC_BitData(BitData* bitData) :
        _bitData(bitData)
    {
    }

    ///
    /// @brief Constructs a bit data value.
    ///
    /// @param size
    ///     The size of the value in bits.
    explicit OpenIPC_BitData(uint64_t size) :
        _bitData(BitData_CreateManaged(size))
    {
    }

    ///
    /// @brief Constructs a bit data value.
    ///
    /// @param size
    ///     The size of the value in bits.
    /// @param value
    ///     The initial value.
    OpenIPC_BitData(uint64_t size, uint64_t value) :
        _bitData(BitData_CreateManagedFromUInt64(size, value))
    {
    }

    ///
    /// @brief Constructs a bit data value from a buffer.
    ///
    /// @param size
    ///     The size of the value in bits.
    /// @param buffer
    ///     A buffer containing the initial value.
    OpenIPC_BitData(uint64_t size, const void* buffer) :
        _bitData(BitData_CreateManagedFromBuffer(size, buffer))
    {
    }

    ///
    /// @brief Constructs a bit data value from a string.
    ///
    /// @param value
    ///     A string containing the initial value.
    explicit OpenIPC_BitData(const std::string& value) :
        _bitData(BitData_CreateManagedFromString(value.c_str()))
    {
    }

    ///
    /// @brief Copies a value.
    ///
    /// @param value
    ///     The value to copy.
    OpenIPC_BitData(const OpenIPC_BitData& value) :
        _bitData(value._bitData ? BitData_CreateManagedFromBitData(value._bitData) : nullptr)
    {
    }

    ///
    /// @brief Moves a value.
    ///
    /// @param value
    ///     The value to move.
    OpenIPC_BitData(OpenIPC_BitData&& value) noexcept :
        _bitData(value._bitData)
    {
        value._bitData = nullptr;
    }

    ///
    /// @brief Frees the bit data if needed.
    virtual ~OpenIPC_BitData()
    {
        if (_bitData)
        {
            BitData_FreeManaged(_bitData);
        }
    }

    ///
    /// @brief Returns a pointer to the underlying C struct.
    BitData* GetCStruct()
    {
        return _bitData;
    }

    ///
    /// @brief Returns a pointer to the underlying C struct.
    const BitData* GetCStruct() const
    {
        return _bitData;
    }

    ///
    /// @brief Returns a pointer to the underlying C struct and removes
    ///        ownership.
    BitData* ExtractCStruct()
    {
        BitData* bitData = _bitData;
        _bitData = nullptr;
        return bitData;
    }

    ///
    /// @brief Gets a pointer to the raw underlying buffer.
    void* GetBuffer()
    {
        assert(_bitData);
        return _bitData->buffer;
    }

    ///
    /// @brief Gets a pointer to the raw underlying buffer.
    const void* GetBuffer() const
    {
        assert(_bitData);
        return _bitData->buffer;
    }

    ///
    /// @brief Gets the size of the value in bits.
    uint64_t GetBitSize() const
    {
        assert(_bitData);
        return _bitData->bitsize;
    }

    ///
    /// @brief Gets the value of a bit at the given offset.
    ///
    /// @param offset
    ///     The offset of the bit.
    ///
    /// @returns True if the bit is 1; false if the bit is 0.
    bool GetBit(uint64_t offset) const
    {
        return BitData_GetBit(_bitData, offset);
    }

    ///
    /// @brief Sets the value of a bit at the given offset.
    ///
    /// @param offset
    ///     The offset of the bit.
    /// @param bit
    ///     The new value of the bit.
    void SetBit(uint64_t offset, bool bit)
    {
        assert(_bitData);
        BitData_SetBit(_bitData, offset, bit);
    }

    ///
    /// @brief Gets the value as an unsigned 32-bit integer.
    uint32_t GetUInt32() const
    {
        return GetUInt32(0);
    }

    ///
    /// @brief Gets the value as an unsigned 32-bit integer.
    ///
    /// @param offset
    ///     The bit offset into the value to start reading the value.
    uint32_t GetUInt32(uint64_t offset) const
    {
        assert(_bitData);
        return BitData_GetUInt32(_bitData, offset);
    }

    ///
    /// @brief Gets the value as an unsigned 64-bit integer.
    uint64_t GetUInt64() const
    {
        return GetUInt64(0);
    }

    ///
    /// @brief Gets the value as an unsigned 64-bit integer.
    ///
    /// @param offset
    ///     The bit offset into the value to start reading the value.
    uint64_t GetUInt64(uint64_t offset) const
    {
        assert(_bitData);
        return BitData_GetUInt64(_bitData, offset);
    }

    ///
    /// @brief Sets the value from an unsigned 32-bit integer.
    ///
    /// @param value
    ///     The new value.
    void SetUInt32(uint32_t value)
    {
        SetUInt32(0, value);
    }

    ///
    /// @brief Sets the value from an unsigned 32-bit integer.
    ///
    /// @param offset
    ///     The bit offset into the value to start writing the value.
    /// @param value
    ///     The new value.
    void SetUInt32(uint64_t offset, uint32_t value)
    {
        assert(_bitData);
        BitData_SetUInt32(_bitData, offset, value);
    }

    ///
    /// @brief Sets the value from an unsigned 64-bit integer.
    ///
    /// @param value
    ///     The new value.
    void SetUInt64(uint64_t value)
    {
        SetUInt64(0, value);
    }

    ///
    /// @brief Sets the value from an unsigned 64-bit integer.
    ///
    /// @param offset
    ///     The bit offset into the value to start writing the value.
    /// @param value
    ///     The new value.
    void SetUInt64(uint64_t offset, uint64_t value)
    {
        assert(_bitData);
        BitData_SetUInt64(_bitData, offset, value);
    }

    ///
    /// @brief Reads the specified bits.
    ///
    /// @param indices
    ///     The indices of the bits to read.
    ///
    /// @returns The resulting value.
    OpenIPC_BitData Read(const OpenIPC_IndexList& indices) const
    {
        assert(_bitData);
        OpenIPC_BitData result(indices.GetLength());
        BitData_ReadIndices(_bitData, indices.GetCStruct(), result.GetCStruct());
        return result;
    }

    ///
    /// @brief Writes the specified bits.
    ///
    /// @param indices
    ///     The indices of the bits to write.
    /// @param value
    ///     The value to write.
    void Write(const OpenIPC_IndexList& indices, const OpenIPC_BitData& value)
    {
        assert(_bitData);
        BitData_WriteIndices(_bitData, indices.GetCStruct(), value.GetCStruct());
    }

    ///
    /// @brief Inserts data inline into the bit data value.
    ///
    /// @param start
    ///     The bit offset into the value to start the insertion after.
    /// @param size
    ///     The number of bits to insert from the source value.
    /// @param value
    ///     The source value to insert from.
    void Insert(uint64_t start, uint64_t size, uint64_t value)
    {
        assert(_bitData);
        BitData_InsertUint64(_bitData, start, size, value);
    }

    ///
    /// @brief Inserts data inline into the bit data value.
    ///
    /// @param start
    ///     The bit offset into the value to start the insertion after.
    /// @param source
    ///     The source value to insert from.
    void Insert(uint64_t start, const OpenIPC_BitData& source)
    {
        assert(_bitData);
        BitData_InsertBitData(_bitData, start, source._bitData, 0, source.GetBitSize());
    }

    ///
    /// @brief Inserts data inline into the bit data value.
    ///
    /// @param start
    ///     The bit offset into the value to start the insertion after.
    /// @param source
    ///     The source value to insert from.
    /// @param sourceStart
    ///     The bit offset into the source value.
    /// @param size
    ///     The number of bits to insert from the source value.
    void Insert(uint64_t start, const OpenIPC_BitData& source, uint64_t sourceStart, uint64_t size)
    {
        assert(_bitData);
        BitData_InsertBitData(_bitData, start, source._bitData, sourceStart, size);
    }

    ///
    /// @brief Appends data to the end of the bit data value.
    ///
    /// @param value
    ///     The value to append.
    void Append(const OpenIPC_BitData& value)
    {
        assert(_bitData);
        BitData_Append(_bitData, value._bitData);
    }

    ///
    /// @brief Deletes a contiguous series of bits.
    ///
    /// @param start
    ///     The bit offset into the value to start deleting.
    /// @param size
    ///     The number of bits to delete.
    void Delete(uint64_t start, uint64_t size)
    {
        assert(_bitData);
        BitData_Delete(_bitData, start, size);
    }

    ///
    /// @brief Resizes the bit data.
    ///
    /// @param size
    ///     The new bit size.
    virtual void Resize(uint64_t size)
    {
        assert(_bitData);
        BitData_Resize(_bitData, size);
    }

    ///
    /// @brief Inverts each bit in the bit data.
    virtual void Invert()
    {
        assert(_bitData);
        BitData_XorValue(_bitData, 0, _bitData->bitsize, 0xffffffffffffffff);
    }

    ///
    /// @brief Performs a bitwise shift-left operation.
    ///
    /// @param count
    ///     The shift count.
    void ShiftLeft(uint64_t count)
    {
        ShiftLeft(0, GetBitSize(), count);
    }

    ///
    /// @brief Performs a bitwise shift-left operation.
    ///
    /// @param start
    ///     The first bit to include in the shift.
    /// @param size
    ///     The number of bits to include in the shift.
    /// @param count
    ///     The shift count.
    void ShiftLeft(uint64_t start, uint64_t size, uint64_t count)
    {
        assert(_bitData);
        BitData_ShiftLeft(_bitData, start, size, count);
    }

    ///
    /// @brief Performs a bitwise shift-right operation.
    ///
    /// @param count
    ///     The shift count.
    void ShiftRight(uint64_t count)
    {
        ShiftRight(0, GetBitSize(), count);
    }

    ///
    /// @brief Performs a bitwise shift-right operation.
    ///
    /// @param start
    ///     The first bit to include in the shift.
    /// @param size
    ///     The number of bits to include in the shift.
    /// @param count
    ///     The shift count.
    void ShiftRight(uint64_t start, uint64_t size, uint64_t count)
    {
        assert(_bitData);
        BitData_ShiftRight(_bitData, start, size, count);
    }

    ///
    /// @brief Performs a bitwise rotate-right operation.
    ///
    /// @param count
    ///     The rotate count.
    void RotateRight(uint32_t count)
    {
        if (count > GetBitSize())
        {
            count %= GetBitSize();
        }
        auto tmp = GetRange(0, count);
        ShiftRight(count);
        SetRange(GetBitSize() - count, count, tmp);
    }

    ///
    /// @brief Gets the value as a binary string.
    std::string ToBinary() const
    {
        return ToBinary(0, GetBitSize());
    }

    ///
    /// @brief Gets the value as a binary string.
    ///
    /// @param start
    ///     The first bit to include in the value.
    /// @param size
    ///     The number of bits to include in the value.
    std::string ToBinary(uint64_t start, uint64_t size) const
    {
        assert(_bitData);
        const size_t capacity = static_cast<size_t>(size + 3);
        std::vector<char> buffer(capacity);
        BitData_ToBinary(_bitData, start, size, &buffer[0], (uint32_t)capacity);
        return std::string(&buffer[0]);
    }

    ///
    /// @brief Gets the value as a hex string.
    std::string ToHex() const
    {
        return ToHex(0, GetBitSize());
    }

    ///
    /// @brief Gets the value as a hex string.
    ///
    /// @param start
    ///     The first bit to include in the value.
    /// @param size
    ///     The number of bits to include in the value.
    std::string ToHex(uint64_t start, uint64_t size) const
    {
        assert(_bitData);
        const size_t capacity = static_cast<size_t>(size / 4 + 4);
        std::vector<char> buffer(capacity);
        BitData_ToHex(_bitData, start, size, &buffer[0], (uint32_t)capacity);
        return std::string(&buffer[0]);
    }

    ///
    /// @brief Gets the value as a decimal string.
    std::string ToDecimal() const
    {
        return ToDecimal(0, GetBitSize());
    }

    ///
    /// @brief Gets the value as a decimal string.
    ///
    /// @param start
    ///     The first bit to include in the value.
    /// @param size
    ///     The number of bits to include in the value.
    std::string ToDecimal(uint64_t start, uint64_t size) const
    {
        assert(_bitData);
        const size_t capacity = static_cast<size_t>(size / 3 + 3);
        std::vector<char> buffer(capacity);
        BitData_ToDecimal(_bitData, start, size, &buffer[0], (uint32_t)capacity);
        return std::string(&buffer[0]);
    }

    ///
    /// @brief Compares the value to another.
    ///
    /// @param value
    ///     The other value to compare to.
    ///
    /// @returns An integer value indicating the relationship between the
    /// values.  A zero value indicates equivalence.  A value greather than
    /// zero indicates that the first value is greater than the other.  A value
    /// less than zero indicates that the first value is less than the other.
    int32_t CompareTo(const OpenIPC_BitData& value) const
    {
        assert(_bitData);
        int32_t result = 0;
        BitData_Compare(_bitData, 0, value._bitData, 0, std::max(GetBitSize(), value.GetBitSize()),  &result);
        return result;
    }

    ///
    /// @brief Assigns a new value.
    ///
    /// @param value
    ///     The new value.
    virtual OpenIPC_BitData& operator=(const OpenIPC_BitData& value)
    {
        if (this != &value)
        {
            if (_bitData)
            {
                BitData_FreeManaged(_bitData);
            }
            _bitData = value._bitData ? BitData_CreateManagedFromBitData(value._bitData) : nullptr;
        }
        return *this;
    }

    ///
    /// @brief Assigns a new value moved from another.
    ///
    /// @param value
    ///     The new value.
    virtual OpenIPC_BitData& operator=(OpenIPC_BitData&& value)
    {
        if (this != &value)
        {
            if (_bitData)
            {
                BitData_FreeManaged(_bitData);
            }
            _bitData = value._bitData;
            value._bitData = nullptr;
        }
        return *this;
    }

    ///
    /// @brief Returns whether the value is equal to another.
    ///
    /// @param value
    ///     The other value.
    bool operator==(const OpenIPC_BitData& value) const
    {
        return CompareTo(value) == 0;
    }

    ///
    /// @brief Returns whether the value is not equal to another.
    ///
    /// @param value
    ///     The other value.
    bool operator!=(const OpenIPC_BitData& value) const
    {
        return CompareTo(value) != 0;
    }

    ///
    /// @brief Returns whether the value is greater than another.
    ///
    /// @param value
    ///     The other value.
    bool operator>(const OpenIPC_BitData& value) const
    {
        return CompareTo(value) > 0;
    }

    ///
    /// @brief Returns whether the value is greater than or equal to another.
    ///
    /// @param value
    ///     The other value.
    bool operator>=(const OpenIPC_BitData& value) const
    {
        return CompareTo(value) >= 0;
    }

    ///
    /// @brief Returns whether the value is less than another.
    ///
    /// @param value
    ///     The other value.
    bool operator<(const OpenIPC_BitData& value) const
    {
        return CompareTo(value) < 0;
    }

    ///
    /// @brief Returns whether the value is less than or equal to another.
    ///
    /// @param value
    ///     The other value.
    bool operator<=(const OpenIPC_BitData& value) const
    {
        return CompareTo(value) <= 0;
    }

    ///
    /// @brief Performs a left shift.
    ///
    /// @param value
    ///     The count of the shift.
    ///
    /// @returns The shifted value.
    OpenIPC_BitData operator<<(uint64_t value) const
    {
        OpenIPC_BitData result(*this);
        result.ShiftLeft(value);
        return result;
    }

    ///
    /// @brief Performs a left shift.
    ///
    /// @param value
    ///     The count of the shift.
    ///
    /// @returns The shifted value.
    OpenIPC_BitData operator<<(const OpenIPC_BitData& value) const
    {
        OpenIPC_BitData result(*this);
        result.ShiftLeft(value.GetUInt64());
        return result;
    }

    ///
    /// @brief Performs a right shift.
    ///
    /// @param value
    ///     The count of the shift.
    ///
    /// @returns The shifted value.
    OpenIPC_BitData operator>>(uint64_t value) const
    {
        OpenIPC_BitData result(*this);
        result.ShiftRight(value);
        return result;
    }

    ///
    /// @brief Performs a right shift.
    ///
    /// @param value
    ///     The count of the shift.
    ///
    /// @returns The shifted value.
    OpenIPC_BitData operator>>(const OpenIPC_BitData& value) const
    {
        OpenIPC_BitData result(*this);
        result.ShiftRight(value.GetUInt64());
        return result;
    }

    ///
    /// @brief Zeros out all the data in the BitData structure.
    void Clear()
    {
        assert(_bitData);
        BitData_InitValue(_bitData, 0, GetBitSize(), 0);
    }

    ///
    /// @brief Initializes a section of the BitData with the value specified.
    ///
    /// @param start
    ///     The bit offset into the value to start the insertion after.
    /// @param size
    ///     The number of bits to insert from the source value.
    /// @param value
    ///     The source value to initialize to. If size is greater than
    ///     64 bits than the value is repeated.
    void Initialize(uint64_t start,	uint64_t size,	uint64_t value)
    {
        assert(_bitData);
        BitData_InitValue(_bitData, start, size, value);
    }

    ///
    /// @brief Gets a section of the value as a OpenIPC_BitData.
    ///
    /// @param start
    ///     The bit offset into the value to start reading the value.
    /// @param size
    ///     The number of bits of the value to retrieve.
    OpenIPC_BitData GetRange(uint64_t start, uint64_t size) const
    {
        assert(_bitData);
        assert(start + size <= _bitData->bitsize);

        OpenIPC_BitData result(size);
        BitData_Copy(_bitData, start, result._bitData, 0, size);
        return result;
    }

    ///
    /// @brief Sets the value from an unsigned 64-bit integer.
    ///
    /// @param offset
    ///     The bit offset into the value to start writing the value.
    /// @param size
    ///     The number of bits of the value to set.
    /// @param value
    ///     The new value.
    void SetUpToUInt64(uint64_t offset, uint32_t size, uint64_t value)
    {
        assert(_bitData);
        BitData_SetUpToUInt64(_bitData, offset, size, value);
    }

    ///
    /// @brief Sets the value from another OpenIPC_BitData.
    ///
    /// @param offset
    ///     The bit offset into the value to start writing the value.
    /// @param size
    ///     The number of bits of the value to set.
    /// @param value
    ///     The new value.
    void SetRange(uint64_t offset, uint32_t size, const OpenIPC_BitData& value)
    {
        assert(_bitData);
        assert(value._bitData);
        BitData_Copy(value._bitData, 0, _bitData, offset, size);
    }

    ///
    /// @brief Sets the value from a BitData structure.
    ///
    /// @param offset
    ///     The bit offset into the value to start writing the value.
    /// @param size
    ///     The number of bits of the value to set.
    /// @param value
    ///     The new value.
    void SetRange(uint64_t offset, uint32_t size, const BitData* value)
    {
        assert(_bitData);
        assert(value);
        BitData_Copy(value, 0, _bitData, offset, size);
    }

    ///
    /// @brief Performs a bitwise AND operation between the value
    ///        in the BitData and an unsigned 64-bit integer.
    ///
    /// @param start
    ///     The bit offset into the BitData's value to start the operation.
    /// @param size
    ///     The number of bits in the range to perform the operation.
    /// @param value
    ///     The value to bitwise AND with BitData's value.
    void BitwiseAnd(uint64_t start,	uint64_t size, uint64_t value)
    {
        assert(_bitData);
        assert(size <= 64);
        assert(start + size <= _bitData->bitsize);

        BitData source = BitData_CreateLocalFromBuffer(64, 64, &value);
        BitData_BitwiseAnd(_bitData, start, _bitData, start, &source, 0, size);
    }

    ///
    /// @brief Performs a bitwise AND operation between the value
    ///        in the BitData and another OpenIPC_BitData value
    ///
    /// @param value
    ///     The value to bitwise AND with BitData's value.
    OpenIPC_BitData& operator&=(const OpenIPC_BitData& value)
    {
        assert(_bitData);

        uint64_t minBitSize = std::min(value.GetBitSize(), GetBitSize());
        BitData_BitwiseAnd(_bitData, 0, _bitData, 0, value._bitData, 0, minBitSize);

        // If the size of this BitData is larger than the one passed in then zero out the upper bits.
        if (GetBitSize() > minBitSize)
        {
            Initialize(minBitSize, (GetBitSize() - minBitSize), 0);
        }

        return (*this);
    }

    ///
    /// @brief Performs a bitwise OR operation between the value
    ///        in the BitData and an unsigned 64-bit integer.
    ///
    /// @param start
    ///     The bit offset into the BitData's value to start the operation.
    /// @param size
    ///     The number of bits in the range to perform the operation.
    /// @param value
    ///     The value to bitwise OR with BitData's value.
    void BitwiseOr(uint64_t start, uint64_t size, uint64_t value)
    {
        assert(_bitData);
        assert(size <= 64);
        assert(start + size <= _bitData->bitsize);

        BitData source = BitData_CreateLocalFromBuffer(64, 64, &value);
        BitData_BitwiseOr(_bitData, start, _bitData, start, &source, 0, size);
    }

    ///
    /// @brief Performs a bitwise OR operation between the value
    ///        in the BitData and another OpenIPC_BitData value
    ///
    /// @param value
    ///     The value to bitwise OR with BitData's value.
    OpenIPC_BitData& operator|=(const OpenIPC_BitData& value)
    {
        assert(_bitData);

        uint64_t minBitSize = std::min(value.GetBitSize(), GetBitSize());
		BitData_BitwiseOr(_bitData, 0, _bitData, 0, value._bitData, 0, minBitSize);
        return (*this);
    }

    ///
    /// @brief Performs a bitwise XOR operation between the value
    ///        in the BitData and an unsigned 64-bit integer.
    ///
    /// @param start
    ///     The bit offset into the BitData's value to start the operation.
    /// @param size
    ///     The number of bits in the range to perform the operation.
    /// @param value
    ///     The value to bitwise XOR with BitData's value.
    void BitwiseXor(uint64_t start,	uint64_t size,	uint64_t value)
    {
        assert(_bitData);
        assert(size <= 64);
        assert(start + size <= _bitData->bitsize);

        BitData source = BitData_CreateLocalFromBuffer(64, 64, &value);
        BitData_BitwiseXor(_bitData, start, _bitData, start, &source, 0, size);
    }

    ///
    /// @brief Performs a bitwise XOR operation between the value
    ///        in the BitData and another OpenIPC_BitData value
    ///
    /// @param value
    ///     The value to bitwise XOR with BitData's value.
    OpenIPC_BitData& operator^=(const OpenIPC_BitData& value)
    {
        assert(_bitData);

        uint64_t minBitSize = std::min(value.GetBitSize(), GetBitSize());
		BitData_BitwiseXor(_bitData, 0, _bitData, 0, value._bitData, 0, minBitSize);
        return (*this);
    }

    ///
    /// @brief Performs a bitwise inversion of the value
    OpenIPC_BitData& operator~()
    {
        assert(_bitData);
        Invert();
        return (*this);
    }

    ///
    /// @brief Returns true if there are no bits set in the value.
    bool IsZero() const
    {
        uint64_t firstBitSet = c_o4NoBitSet;
        BitData_FindFirst(_bitData, 0, _bitData->bitsize, &firstBitSet);
        return firstBitSet == c_o4NoBitSet;
    }

private:
    BitData* _bitData;
};

inline std::ostream& operator<< (std::ostream& os, const OpenIPC_BitData& value)
{
    os << value.ToHex();
    return os;
}
