/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief Implementation of functions for creating, destroying, and
///         manipulating IndexList objects exposed as C-style functions.
///
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include "IndexListTypes.h"

#include <Foundation/Types.h>

#if defined(HOST_WINDOWS)
    #if defined(INDEXLIST_EXPORTS)
         #define EXPORT_INDEXLIST_API __declspec(dllexport)
    #elif defined(INDEXLIST_IMPORTS)
        #define EXPORT_INDEXLIST_API __declspec(dllimport)
    #else
        #define EXPORT_INDEXLIST_API
    #endif
#else
    #if defined(INDEXLIST_EXPORTS) || defined(INDEXLIST_IMPORTS)
        #define EXPORT_INDEXLIST_API __attribute__ ((visibility("default")))
    #else
        #define EXPORT_INDEXLIST_API
    #endif
#endif

extern "C"
{
    ///
    /// @brief Creates an an empty list.
    ///
    /// @returns The newly created index list.
    EXPORT_INDEXLIST_API IndexList* IndexList_Create(
        void
    );

    ///
    /// @brief Creates a new index list from a string.
    ///
    /// @param string The string to create the index list from.
    ///
    /// @returns The newly created index list.
    EXPORT_INDEXLIST_API IndexList* IndexList_CreateFromString(
        const char* string
    );

    ///
    /// @brief Destroys an index list.
    ///
    /// @param indexList The index list to destroy.
    EXPORT_INDEXLIST_API void IndexList_Free(
        IndexList*  indexList
    );

    ///
    /// @brief Moves an index list iterator to the next value.
    ///
    /// If the iterator is initialized to zero, the first value of the index
    /// list will be moved to.
    ///
    /// @param indexList The index list.
    /// @param iterator The iterator to move to the next value.
    ///
    /// @returns True if a new value was moved to; false if the end of the
    ///          index has been reached.
    EXPORT_INDEXLIST_API bool IndexList_MoveNext(
        const IndexList*    indexList,
        IndexListIterator*  iterator
    );

    ///
    /// @brief Appends a new index to the end of an index list.
    ///
    /// @param indexList The index list.
    /// @param index The index to append.
    EXPORT_INDEXLIST_API void IndexList_Append(
        IndexList*  indexList,
        uint64_t     index
    );

    ///
    /// @brief Appends a contiguous range of indices to the end of an index
    ///        list.
    ///
    /// @note The index range may be increasing or decreasing.
    ///
    /// @param indexList The index list.
    /// @param start The start of the index range to append.
    /// @param end The end of the index range to append.
    EXPORT_INDEXLIST_API void IndexList_AppendRange(
        IndexList*  indexList,
        uint64_t     start,
        uint64_t     end
    );

    ///
    /// @brief Applies an offset to each index in an index list.
    ///
    /// @param indexList The index list.
    /// @param offset The offset to apply to each index.
    EXPORT_INDEXLIST_API void IndexList_ApplyOffset(
        IndexList*  indexList,
        uint64_t     offset
    );

    ///
    /// @brief Slices an index list by another index list.
    ///
    /// @param indexList The index list.
    /// @param slice The index list used to slice.
    ///
    /// @returns The sliced index list.
    EXPORT_INDEXLIST_API IndexList* IndexList_Slice(
        IndexList*  indexList,
        IndexList*  slice
    );

    ///
    /// @brief Resizes an index list to a new length.
    ///
    /// @param indexList The index list.
    /// @param length The new length of the index list.
    EXPORT_INDEXLIST_API void IndexList_Resize(
        IndexList*  indexList,
        size_t      length
    );

    ///
    /// @brief Gets the index at the specified position of an index list.
    ///
    /// @param indexList The index list.
    /// @param position The position of the index to get.
    ///
    /// @returns The index at the specified position; -1 if no index exists at
    ///          the specified position.
    EXPORT_INDEXLIST_API uint64_t IndexList_GetIndex(
        const IndexList*    indexList,
        size_t              position
    );

    ///
    /// @brief Gets maximum index in an index list.
    ///
    /// @param indexList The index list.
    ///
    /// @returns The maximum index; 0 if the index list is empty.
    EXPORT_INDEXLIST_API uint64_t IndexList_GetMaxIndex(
        const IndexList*    indexList
    );

    ///
    /// @brief Gets minimum index in an index list.
    ///
    /// @param indexList The index list.
    ///
    /// @returns The minimum index; 0 if the index list is empty.
    EXPORT_INDEXLIST_API uint64_t IndexList_GetMinIndex(
        const IndexList*    indexList
    );

    ///
    /// @brief Gets the length of an index list.
    ///
    /// @param indexList The index list.
    EXPORT_INDEXLIST_API size_t IndexList_GetLength(
        const IndexList*    indexList
    );

    ///
    /// @brief Attempts to extract the offset and length of the index list.
    ///
    /// @param indexList The index list.
    /// @param offset The offset of the index list.
    /// @param length The length of the index list.
    ///
    /// @returns True if the index list is a simple offset and length; false
    ///          otherwise.
    EXPORT_INDEXLIST_API bool IndexList_ExtractOffsetAndLength(
        const IndexList*    indexList,
        uint64_t*           offset,
        size_t*             length
    );

    ///
    /// @brief Returns how many characters are needed to represent an index
    ///        list as a string.
    ///
    /// @param indexList The index list.
    EXPORT_INDEXLIST_API size_t IndexList_GetStringSize(
        const IndexList*    indexList);

    ///
    /// @brief Converts an index list to string.
    ///
    /// @param indexList The index list.
    /// @param buffer The bufer to write the string to.
    /// @param bufferSize The size of the buffer.
    EXPORT_INDEXLIST_API void IndexList_ToString(
        const IndexList*    indexList,
        char*               buffer,
        size_t              bufferSize);
}
