/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#pragma once

#include <string>

#pragma once

#if defined(HOST_WINDOWS)
#   if defined(TELEMETRY_EXPORTS)
        #       define EXPORT_TELEMETRY_API __declspec(dllexport)
#   else
        #       define EXPORT_TELEMETRY_API __declspec(dllimport)
#   endif
#else
    #   define EXPORT_TELEMETRY_API __attribute__ ((visibility("default")))
#endif

#define TELEMETRY_BUILD_DATA(eventname, keyname, datastream)                    \
    {                                                                                                                                               \
        std::stringstream stream;                                                                                       \
        stream << datastream;                                                                                           \
        std::string eventstring = stream.str();                                                         \
        Telemetry_BuildDataSet(eventname, keyname, eventstring.c_str(), false); \
    }

#define TELEMETRY_BUILD_DATA_ENCRYPTED(eventname, keyname, datastream)  \
    {                                                                                                                                               \
        std::stringstream stream;                                                                                       \
        stream << datastream;                                                                                           \
        std::string eventstring = stream.str();                                                         \
        Telemetry_BuildDataSet(eventname, keyname, eventstring.c_str(), true);  \
    }

#include "TelemetryStringIDs.h"

extern "C"
{
    /*!
    \brief initialize the telemetry implementation

    \param appname the name of the client application
    \param appver the version of the application, in string form

    OpenIPC should call this API once per process lifetime
    */
    EXPORT_TELEMETRY_API void Telemetry_Initialize(const char* appname, const char* appver);

    /*!
    \brief deinitialize the telemetry implementation

    OpenIPC should call this once per process lifetime, either in the normal shutdown path
    or in a panic handler
    */
    EXPORT_TELEMETRY_API void Telemetry_Deinitialize();

    /*!
    \brief begin a named session

    \param sessionname the name of the session

    This call will initialize a new session the supplied name.  Subsequent calls to the event APIs can supply this session
    name in order to associate those events with that session.
    */
    EXPORT_TELEMETRY_API void Telemetry_BeginSession(const char* sessionname);

    /*!
    \brief end a named session

    \param sessionname the name of the session
    */
    EXPORT_TELEMETRY_API void Telemetry_EndSession(const char* sessionname);

    /*!
    \brief send a telemetry event, optionally associated with a session

    \param sessionname The optional session to associate this event with
    \param eventname The name of the event

    This is the primary API for emitting telemetry events, it will immediately emit an event with the specified
    name, and optionally associate it with the supplied session name.  To add key:val pairs to this event you can
    call the Telemetry_BuildDataSet function *prior* to calling this function.
    */
    EXPORT_TELEMETRY_API void Telemetry_SendEvent(const char* sessionname, const char* eventname);

    /*!
    \brief Add a key:value pair to a dataset, which will be later be attached to an event

    \param eventname the name of the event to which key:value data should be attached
    \param key the key for the key:value
    \param value the value for the key:value
    \param encrypt if true then encrypt the data (but not the key!)

    This API is provided so that an arbitrary component in OpenIPC can contribute to an event that is
    ultimately created by a different component.  This is necessary because the telemetry requirements are orthogonal
    to the OpenIPC software architecture, and we dont want to be passing "event handles" all over the place...

    To use this API, a component simply provides a key:value pair, and the name of the event this should be associated with.
    The next time an event with this name is emitted (by any component) the queued data is appended, and the dataset is flushed.

    The usage of this API is conventional and the contributions from components to an event must be managed
    by the developer.
    */
    EXPORT_TELEMETRY_API void Telemetry_BuildDataSet(const char* eventname, const char* key, const char* value, bool encrypt);

    /*!
    \brief return true if there is data present in a dataset

    \param eventname the name of the event for which the test should be performed

    This API is used to check if any subordinate code has contributed to an event's dataset, e.g. higher-level code can elect 
    *not* to emit an event if subordinate code has not contributed anything
    */
    EXPORT_TELEMETRY_API bool Telemetry_HasDataSet(const char* eventname);

    /*!
    \brief discard data that was built with BuildDataSet

    \param eventname the name of the event for which accrued data should be discarded

    This API is used to discard data that was accrued via the BuildDataSet API in the case where the associated event will *not* be emitted.
    For example, the software may accrue data related to an operation that will be relevant later in the case of a failure, but is not interesting
    if a failure does not occur.
    */
    EXPORT_TELEMETRY_API void Telemetry_DiscardDataSet(const char* eventname);

    /*!
    \brief upload a file to the Telemetry DB

    \param sessionname the session to which the file should be associated (optional)
    \param eventname the event to which the filename and upload GUID should be appended (optional)
    \param friendlyfilename a human-readable nam e for the file, which will be visible in the database
    \param filepath the path to the file for upload

    This API can be used to upload arbitrary data, e.g. a log file or config file

    The file will be copied by the telemetry framework to a temp folder for upload, so once this API call returns the original file
    can be modified or deleted.

    If the eventname parameter is supplied, then the GUID generated during the upload process is appended to that event.  This facilitates
    correlation of an event (in the events table) with a file upload (in the uploads table)
    */
    EXPORT_TELEMETRY_API void Telemetry_UploadFile(const char* sessionname, const char* eventname, const char* friendlyfilename, const char* filepath);

    /*!
    \brief records an event to be sent as an aggregate count across the session.

    \param sessionname The optional session to associate this event with
    \param eventname the name of the event to which key:value data should be attached
    \param key the key for the key:value
    \param value the value for the key:value

    This API can be used to count events that are aggregated over time.

    The count will be tracked for each combination of sessionname, eventname, key, and value. When Telemetry_EndSession or Telemetry_Deinitialize
    is called the events tracked by this call will be sent.
    */
    EXPORT_TELEMETRY_API void Telemetry_CountEvent(const char* sessionname, const char* eventname, const char* key, const char* value);
};
