#
#  INTEL CONFIDENTIAL
#
#  Copyright 2008-2013 Intel Corporation All Rights Reserved.
#
#  The source code contained or described herein and all documents related to
#  the source code ("Material") are owned by Intel Corporation or its
#  suppliers or licensors.
#
#  Title to the Material remains with Intel Corporation or its suppliers and
#  licensors. The Material contains trade secrets and proprietary and
#  confidential information of Intel or its suppliers and licensors.
#  The Material is protected by worldwide copyright and trade secret laws and
#  treaty provisions. No part of the Material may be used, copied, reproduced,
#  modified, published, uploaded, posted, transmitted, distributed,
#  or disclosed in any way without Intel's prior express written permission.
#
#  No license under any patent, copyright, trade secret or other intellectual
#  property right is granted to or conferred upon you by disclosure
#  or delivery of the Materials, either expressly, by implication, inducement,
#  estoppel or otherwise. Any license under such intellectual property rights
#  must be express and approved by Intel in writing.
#

. ./config.sh

#-----------------------------------------------------------------------------
#   Function:  Print
#
#   Description:
#  Function filters messages. If the debug level of the message is below the current DEBUG_LEVEL parameter value the message will not be printed.
#
#   Parameters:
#      $1 - Message to print
#      $2 - Debug level of the current message
#      $3 - Color of the message. Colors codes:
#                       31 - Red
#                       32 - Green
#                       34 - Blue
#--------------------------------------------------------------------------
Print() {
  if [ "$DEBUG_LEVEL" -ge "$2" ]; then
    if [ "$3" != "" ]; then
      echo -e "\E[${3};40m${1}\E[40;37m"
    else
      echo -e "$1"
    fi
  fi
}

if [ -f "$ipmitoolPath" ]
then
  Print "Ipmitool found." 10 32
else
  Print "Ipmitool not found please check your path in config file" 0 31
   exit
fi

IPMITOOL="$ipmitoolPath -I lanplus -H $BMC_IP -U $USERNAME -P $PASSWORD -b 0x06 -C 3 -t $ME_ADDRESS"
suppressCmdError=0


#-----------------------------------------------------------------------------
#   Function:  SetTargetAddr
#
#   Description:
#   Function sets target device address
#
#   Parameters:
#      $1 - target device address
#-------------------------------------
SetTargetAddr() {
IPMITOOL="$ipmitoolPath -I lanplus -H $BMC_IP -U $USERNAME -P $PASSWORD -b 0x06 -C 3 -t $1"
}

ParseTestArguments () {
  if [ "$#" -gt 0 ]; then
    DEBUG_LEVEL=$1
  fi
}

#-----------------------------------------------------------------------------
#   Function:  Command
#
#   Description:
#  Function sends  the IPMI raw command  using the IPMItool
#
#   Parameters:
#      $1 - IPMI raw command to send
#
#   Returns:
#      scriptError - error flag. Sygnalizes that an error occured during execution of any IPMI command in the whole script. Should be cleared at the beginning and at the end of the srcipt.
#      error - error flag. Sygnalizes that an error occured during execution of single IPMI command.  The flag is automatically cleared after successful sending of the IPMI command.
#---------------------------------------------------------------------------
Command() {
  Print " Run command: $1" 10
  local i=0
  errorCode=
  while [ $i -lt 5 ]; do
    i=$(( $i + 1 ))
    Print "  Attempt: $i" 10
    rsp=`$IPMITOOL $1 2>&1`
    if [ "$?" -gt 0 ]; then
      error=1
      ccPattern="0x[cC]0"
      if [[ "$rsp" =~ $ccPattern ]]; then
      Print "  Completion code: ${BASH_REMATCH[0]} (Node Busy)" 0 31
      continue
    fi
    if [ $suppressCmdError -eq 1 ]; then
      errorCode=`echo -e "$rsp" | sed -ne 's/^.*rsp=0x//p' | sed -n 's/):.*//p'`
      return
    fi
    break
    else
      rsp=$(echo -e "$rsp" | sed -e 's%\r\n%\n%g')
      Print "  Command response: $rsp " 10
      error=0
      errorCode="00"
      break
    fi
  done
  if [ $error -eq 1 ]; then
    scriptError=1
    Print "  Error: $rsp" 0 31
    rsp=""
  fi
}

SendIpmiCmd() {
  Print " Sending the IPMI command: $1" 9
  Command "raw $1"
  if [ $error -eq 0 ]; then
  Print " IPMI response: $rsp" 9
  fi
}

#-----------------------------------------------------------------------------
#   Function:  SendNmCmd
#
#   Description:
#  Function adds NM_NET_FUNC to the IPMI raw command  amd invokes the SendIpmiCmd function
#
#   Parameters:
#      IPMI raw command
#
#---------------------------------------------------------------------------
SendNmCmd() {
  Print " Sending the NM command: $1" 8
  SendIpmiCmd "$NM_NET_FUNC $1 0x57 0x01 0x00 $2"
  if [ "$error" -eq 0 ]; then
  Print " NM response: $rsp" 8
  fi
}

SendPeciCmd() {
  Print " Send Peci command: $1" 8
  SendIpmiCmd "$PECI_NET_FUNC $1 $2"
  if [ "$error" -eq 0 ]; then
  Print " PECI response: $rsp" 8
  fi
}

#-----------------------------------------------------------------------------
#   Function:  SendDiagCmdProxy
#
#   Description:
#  Function sends Diagnostic Command through Ipmi Cmd
#
#   Parameters:
#       $1 - Diagnostic command
#
#---------------------------------------------------------------------------
SendDiagCmdProxy() {
  Print " Sending Diagnostic Command: $1" 8
  SendIpmiCmd "0x30 0x26 0x57 0x01 0x00 0x04 $1"
  if [ "$error" -eq 0 ]; then
    read r1 r2 r3 r4 rest <<< $rsp
    Print " Diagnostic response: $rest" 8
  fi
}


#-----------------------------------------------------------------------------
#   Function:  checkArguments
#
#   Description:
#  Function  checks if passed arguments are equal
#
#   Parameters:
#      $1 - set arguments
#      $2 - get arguments
#
#   Returns:
#     isEqual :
#    1- Arguments matches
#    0- Arguments does not match
#
#-----------------------------------------------------------------------------
checkArguments(){
  local -a array1=(${1})
  local -a array2=(${2})
  local text1=${array1[@]}
  local text2=${array2[@]}
  Print "Array 1:  $text1" 10
  Print "Array 2:  $text2" 10
  isEqual=1
  if [ ${#array1[@]} -ne ${#array2[@]} ]; then
    Print " Error during comparision: " 6
    Print "   size of array 1: ${#array1[@]}" 6
    Print "   size of array 2: ${#array2[@]} " 6
    Print " Error during execution checkArguments function: Array 1 have different size than array 2" 0 31
    isEqual=0
  else
    local ArrayLength=${#array1[@]}
    while [ $ArrayLength -gt 0 ]
    do
      let "ArrayLength = $ArrayLength - 1"
      if [ $(( ${array1[$ArrayLength]} )) -ne $(( ${array2[$ArrayLength]} )) ]; then
        Print " Error during execution checkArguments function: Array[$ArrayLength] $(( ${stab[$ArrayLength]} )) is not equal to $(( ${tab[$ArrayLength]} ))  " 0 31
        isEqual=0
        break
      else
        Print " Array[$ArrayLength] $(( ${stab[$ArrayLength]} )) is equal to $(( ${tab[$ArrayLength]} ))  " 10
      fi
    done
  fi
}

#-----------------------------------------------------------------------------
#   Function:  EnDisNMPolicyControl
#
#   Description:
#  Function executes the  Enable/Disable Node Manager Policy Control command
#
#   Command code: C0h
#-----------------------------------------------------------------------------
EnDisNMPolicyControl(){
  Print "Sending the Enable/Disable Node Manager Policy Control command" 5 32
  Print " Enable/Disable Node Manager Policy Control request:" 7 36
  read reqB4 reqB5 reqB6 <<< $1
  # Decoding Flags parameter
  reqB4=$(($reqB4))
  case $reqB4 in
    0)
      Print "   Flags: Global Disable Node Manager policy control" 7
      ;;
    1)
      Print "   Flags: Global Enable Node Manager policy control " 7
      ;;
    2)
      Print "   Flags: Per Domain Disable Node Manager policy control " 7
      ;;
    3)
      Print "   Flags: Per Domain Enable Node Manager policy control " 7
      ;;
    4)
      Print "   Flags: Per Policy Disable Node Manager policy control " 7
      ;;
    5)
      Print "   Flags: Per Policy Enable Node Manager policy control " 7
      ;;
    *)
      Print "   Flags: Unknown($reqB4)" 7
      ;;
  esac
  Print "   DomainID: $reqB5" 7
  Print "   PolicyID: $reqB6" 7
  Print "" 7
  SendNmCmd "0xC0" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Enable/Disable Node Manager Policy Control response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  else
    Print " The Enable/Disable Node Manager Policy Control command returned an error" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetNMPolicy
#
#   Description:
#  Function executes the  Set Node Manager Policy command
#
#   Command code: C1h
#-----------------------------------------------------------------------------
SetNMPolicy(){
  Print "Sending the Set Node Manager Policy command" 5 32
  Print " Set Node Manager Policy request:" 7 36
  read reqB4 reqB5 reqB6 reqB7 reqB8 reqB9 reqB10 reqB11 reqB12 reqB13 reqB14 reqB15 reqB16 reqB17 <<< $1
  Print "   DomainID: $(($reqB4 & 0xF))" 7
  if [ $(( 0x10 & $(($reqB4)) )) = $((0x10)) ];then
    Print "   Policy Enabled: enabled" 7
  else
    Print "   Policy Enalbed: disabled" 7
  fi
  Print "   PolicyID: $reqB5" 7
  if [ $(( 0xF & $(($reqB6)) )) = 0 ];then
    Print "   Policy Trigger Type: No Policy Trigger ( In that case Policy Trigger Limit should be ignored)" 7
  else
    Print "   Policy Trigger Type: Inlet Temperature Limit Policy Trigger in [Celsius]" 7
  fi
  if [ $(( 0x10 & $(($reqB6)) )) = $((0x10)) ];then
    Print "   Configuration Action: Adding Power Policy" 7
  else
    Print "   Configuration Action: Removing Power Policy" 7
  fi

  Print "   Policy Exception Actions:" 7
  if [ $(( $(($reqB7)) & 0x01 )) = $((0x01)) ]; then
      Print "      send alert enabled " 7
  else
    Print "      send alert disabled " 7
  fi
  if [ $(( $(($reqB7)) & 0x02 )) = $((0x02)) ]; then
      Print "      shutdown system enabled " 7
  else
      Print "      shutdown system disabled " 7
  fi
  local reqPowerLimit=$(($reqB9*256+$reqB8))
  Print "   Power Limit: $reqPowerLimit" 7
  local reqCorrTime=$(($reqB13*16777216 + $reqB12*65536 + $reqB11*256 + $reqB10))
  Print "   Correction Time Limit : $reqCorrTime" 7
  local reqTriggerLimit=$(($reqB15*256 + $reqB14))
  Print "   Policy Trigger Limit  : $reqTriggerLimit" 7
  local reqStatRepPeriod=$(($reqB17*256 + $reqB16))
  Print "   Statistics Reporting Period : $reqStatRepPeriod s" 7
  Print "" 7
  
  SendNmCmd "0xC1" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Node Manager Policy response:" 6
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMPolicy
#
#   Description:
#  Function executes the  Get Node Manager Policycommand
#
#   Command code: C2h
#-----------------------------------------------------------------------------
GetNMPolicy(){
  Print "Sending the Get Node Manager Policy command" 5 32
  Print " Get Node Manager Policy request:" 7 36
  read reqB4 reqB5 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "   PolicyID: $reqB5" 7
  Print "" 7

  SendNmCmd "0xC2" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Policy response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   Domain Id: $(( $((0x$b5)) & $((0x0F)) ))" 6
    if [ $(( $((0x$b5)) &  0x10 )) = $((0x10)) ]; then
      Print "   Policy enabled" 6
    else
      Print "   Policy disabled" 6
    fi
    if [ $(( 0x20 & $((0x$b5)) )) = $((0x20)) ]; then
      Print "   per Domain Node Manager policy control enabled " 6
    else
      Print "   per Domain Node Manager policy control disabled " 6
    fi
    if [ $(( 0x40 & $((0x$b5)) )) =  $((0x40)) ]; then
      Print "   Global Node Manager policy control enabled " 6
    else
      Print "   Global Node Manager policy control disabled " 6
    fi
    if [ $(( $(( 0x0F)) & $((0x$b6)) )) = 1 ]; then
      Print "   Policy Trigger Type: Inlet Temperature Limit Policy Trigger in [Celsius]" 6
    else
      Print "   Policy Trigger Type: No Policy Trigger. Policy will maintain Power Limit (In that case Policy Trigger Value will be equal to the Power Limit)" 6
    fi
    if [ $(( $((0xf0)) & $((0x$b6)) )) = $((0x10)) ]; then
      Print "   Policy Type: Power Control Policy" 6
    else
      Print "   Policy Type: Unknown" 6 31
    fi
    Print "   Policy Exception Actions:" 6
    if [ $(( $((0x$b7)) & 0x01 )) = $((0x01)) ]; then
      Print "      send alert enabled " 6
    else
      Print "      send alert disabled " 6
    fi
    if [ $(( $((0x$b7)) & 0x02 )) = $((0x02)) ]; then
      Print "      shutdown system enabled " 6
    else
      Print "      shutdown system disabled " 6
    fi
    Print "   Power Limit: $(( 0x$b9$b8 )) [Watts] " 6
    Print "   Print Correction Time Limit: $(( 0x${b13}${b12}${b11}${b10} )) [ms]" 6
    if [ $(( $((0x$b6)) & 0x0F )) = 0 ]; then
      Print "   Policy Trigger Limit: $(( 0x${b15}${b14} )) [Watts]" 6
    else
      Print "   Policy Trigger Limit: $(( 0x${b15}${b14} )) [Celcius]" 6
    fi
    Print "   Statistics Reporting Period: $(( 0x${b17}${b16} )) [s]" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetNMAlertThresholds
#
#   Description:
#  Function executes the  Set Node Manager Alert Thresholds command
#
#   Command code: C3h
#-----------------------------------------------------------------------------
SetNMAlertThresholds() {
  Print "Sending the Set Node Manager Alert Thresholds command" 5 32
  Print " Set Node Manager Alert Thresholds request:" 7 36
  read reqB4 reqB5 reqB6 reqB7 reqB8 reqB9 reqB10 reqB11 reqB12 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "   PolicyID: $(($reqB5))" 7
  Print "   Number of alert thresholds: $(($reqB6))" 7
  local AlertThreshold=0
  if [ $(($reqB6)) -gt  0 ];then
    AlertThreshold=$(($reqB8*256 + $reqB7))
    Print "   Alert threshold 1: $AlertThreshold" 7
  fi
  if [ $(($reqB6)) -gt  1 ];then
    AlertThreshold=$(($reqB10*256 + $reqB9))
    Print "   Alert threshold 2: $AlertThreshold" 7
  fi
  if [ $(($reqB6)) -gt  2 ];then
    AlertThreshold=$(($reqB12*256 + $reqB11))
    Print "   Alert threshold 3: $AlertThreshold" 7
  fi
  Print "" 7

  SendNmCmd "0xC3" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Node Manager Alert Thresholds response: " 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMAlertThresholds
#
#   Description:
#  Function executes the  Get Node Manager Alert Thresholds command
#
#   Command code: C4h
#-----------------------------------------------------------------------------
GetNMAlertThresholds() {
  Print "Sending the Get Node Manager Alert Thresholds command" 5 32
  Print " Get Node Manager Alert Thresholds request:" 7 36
  read reqB4 reqB5 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "   PolicyID: $(($reqB5))" 7
  Print "" 7

  SendNmCmd "0xC4" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Alert Thresholds response: " 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   Number of alert thresholds: $(( 0x$b5 ))" 6
    if [ $((0x$b5)) -gt  0 ];then
      Print "   Alert threshold 1: $(( 0x$b7$b6 ))" 6
    fi
    if [ $((0x$b5)) -gt  1 ];then
      Print "   Alert threshold 2: $(( 0x$b9$b8 ))" 6
    fi
    if [ $((0x$b5)) -gt  2 ];then
      Print "   Alert threshold 3: $(( 0x$b11$b10 ))" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetNMPolicySuspendPeriods
#
#   Description:
#  Function executes the  Set Node Manager Policy Suspend Periods command
#
#   Command code: C5h
#-----------------------------------------------------------------------------
SetNMPolicySuspendPeriods() {
  Print "Sending the Set Node Manager Policy Suspend Periods command" 5 32
  Print " Set Node Manager Policy Suspend Periods request:" 7 36
  read reqB4 reqB5 reqB6 reqB7 reqB8 reqB9 reqB10 reqB11 reqB12 reqB13 reqB14 reqB15 reqB16 reqB17 reqB18 reqB19 reqB20 reqB21 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "   PolicyID: $reqB5" 7
  Print "   Number of alert thresholds: $(($reqB6))" 7
  if [ $(($reqB6)) -gt 0 ];then
    local tab=( $reqB7 $reqB8 $reqB9 $reqB10 $reqB11 $reqB12 $reqB13 $reqB14 $reqB15 $reqB16 $reqB17 $reqB18 $reqB19 $reqB20 $reqB21)
    local temp=0
    local index=0
    while [ $temp -lt $(($reqB6)) ];do
      Print "   Policy suspend period $(( temp + 1 )): " 7
      index=$(( $temp * 3 ))
      Print "     Policy suspend start time encoded as a number of minutes from mid-night divided by 6: $((${tab[$index]})) " 7
      index=$(( $index + 1 ))
      Print "     Policy suspend stop time encoded as a number of minutes from mid-night divided by 6:  $((${tab[$index]})) " 7
      index=$(( $index + 1 ))
      if [ $(( $((${tab[$index]} )) & 0x40 )) = $((0x40)) ]; then
        Print "     Repeat the suspend period every Sunday" 7
      fi
      if [ $(( $(( ${tab[$index]} )) & 0x20 )) = $((0x20)) ]; then
        Print "     Repeat the suspend period every Saturday" 7
      fi
      if [ $(( $(( ${tab[$index]} )) & 0x10 )) = $((0x10)) ]; then
        Print "     Repeat the suspend period every Friday" 7
      fi
      if [ $(( $(( ${tab[$index]} )) & 0x08 )) = $((0x08)) ]; then
        Print "     Repeat the suspend period every Thursday" 7
      fi
      if [ $(( $(( ${tab[$index]} )) & 0x04 )) = $((0x04)) ]; then
        Print "     Repeat the suspend period every Wednesday" 7
      fi
      if [ $(( $(( ${tab[$index]} )) & 0x02 )) = $((0x02)) ]; then
        Print "     Repeat the suspend period every Tuesday" 7
      fi
      if [ $(( $(( ${tab[$index]} )) & 0x01 )) = $((0x01)) ]; then
        Print "     Repeat the suspend period every Monday" 7
      fi
      temp=$(( $temp + 1 ))
    done
  fi
  Print "" 7

  SendNmCmd "0xC5" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Node Manager Policy Suspend Periods response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMPolicySuspendPeriods
#
#   Description:
#  Function executes the  Get Node Manager Policy Suspend Periods command
#
#   Command code: C6h
#-----------------------------------------------------------------------------
GetNMPolicySuspendPeriods() {
  Print "Sending the Get Node Manager Policy Suspend Periods command" 5 32
  Print " Get Node Manager Policy Suspend Periods request:" 7 36
  read reqB4 reqB5 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "   PolicyID: $reqB5" 7
  Print "  " 7

  SendNmCmd "0xC6" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Policy Suspend Periods response:" 6
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   Number of policy suspend periods: $(( 0x$b5 ))" 6
    if [ $((0x$b5)) -gt  0 ];then
      tab=( $b6 $b7 $b8 $b9 $b10 $b11 $b12 $b13 $b14 $b15 $b16 $b17 $b18 $b19 $b20)
      temp=0
      while [ $temp -lt $b5 ];do
        Print "   Policy suspend period $(( temp + 1 )): " 6
        index=$(( $temp * 3 ))
        Print "     Policy suspend start time encoded as a number of minutes from mid-night divided by 6: $(( 0x${tab[$index]} )) " 6
        index=$(( $index + 1 ))
        Print "     Policy suspend stop time encoded as a number of minutes from mid-night divided by 6:  $(( 0x${tab[$index]} ))" 6
        index=$(( $index + 1 ))
        if [ $(( 0x${tab[$index]} & 0x40 )) = $((0x40)) ]; then
          Print "     Repeat the suspend period every Sunday" 6
        fi
        if [ $(( 0x${tab[$index]} & 0x20 )) = $((0x20)) ]; then
          Print "     Repeat the suspend period every Saturday" 6
        fi
        if [ $(( 0x${tab[$index]} & 0x10 )) = $((0x10)) ]; then
          Print "     Repeat the suspend period every Friday" 6
        fi
        if [ $(( 0x${tab[$index]} & 0x08 )) = $((0x08)) ]; then
          Print "     Repeat the suspend period every Thursday" 6
        fi
        if [ $(( 0x${tab[$index]} & 0x04 )) = $((0x04)) ]; then
          Print "     Repeat the suspend period every Wednesday" 6
        fi
        if [ $(( 0x${tab[$index]} & 0x02 )) = $((0x02)) ]; then
          Print "     Repeat the suspend period every Tuesday" 6
        fi
        if [ $(( 0x${tab[$index]} & 0x01 )) = $((0x01)) ]; then
          Print "     Repeat the suspend period every Monday" 6
        fi
        temp=$(( $temp + 1 ))
      done
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  ResetNMStatistics
#
#   Description:
#  Function executes the  Reset Node Manager Statistics command
#
#   Command code: C7h
#-----------------------------------------------------------------------------
ResetNMStatistics() {
  Print "Sending the Reset Node Manager Statistics command" 5 32
  Print " Reset Node Manager Statistics request:" 7 36
  read reqB4 reqB5 reqB6 <<< $1
  case $(($reqB4)) in
    0)
      Print "   Mode: reset global statistics including power statistics and inlet temperature statistics" 7
      ;;
    1)
      Print "   Mode: per policy statistics including power and trigger statistics " 7
      ;;
  esac
  Print "   DomainID: $reqB5" 7
  Print "   PolicyID: $reqB6" 7
  Print "" 7

  SendNmCmd "0xC7" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Reset Node Manager Statistics response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMStatistics
#
#   Description:
#  Function executes the  Get Node Manager Statistics command
#
#   Command code: C8h
#-----------------------------------------------------------------------------
GetNMStatistics() {
  Print "Sending the Get Node Manager Statistics command" 5 32
  Print " Gst Node Manager Statistics request:" 7 36
  read reqB4 reqB5 reqB6 <<< $1
  case $(($reqB4)) in
    1)
      Print "   Mode: global power statistics in [Watts]" 7
      ;;
    2)
      Print "   Mode: global inlet temperature statistics in [Celsius]" 7
      ;;
    17)
      Print "   Mode: per policy power statistics in [Watts]" 7
      ;;
    18)
      Print "   Mode: per policy trigger statistics in [Clesius]" 7
      ;;

  esac
  Print "   DomainID: $reqB5" 7
  Print "   PolicyID: $reqB6" 7
  Print "" 7

  SendNmCmd "0xC8" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Statistics response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi

    currentvalue="$b6$b5"
    minvalue="$b8$b7"
    maxvalue="$b10$b9"
    avvalue="$b12$b11"
    timestamp="$b16$b15$b14$b13"
    stat="$b20$b19$b18$b17"

    Print "   Current: $(( 0x$b6$b5 ))" 6
    Print "   Minimum: $(( 0x$b8$b7 ))" 6
    Print "   Maximum: $(( 0x$b10$b9 ))" 6
    Print "   Average: $(( 0x$b12$b11 ))" 6
    Print "   Timestamp: $(( 0x$b16$b15$b14$b13 ))" 6
    Print "   Statistics Reporting Period: $(( 0x$b20$b19$b18$b17 )) s" 6
    Print "   domain Id: $(( $((0x$b21))& 0x0F ))" 6
    if [ $(( $((0x$b21)) & 0x10)) = $((0x10)) ];then
      Print "   Policy/Global Administrative state: 1" 6
    else
      Print "   Policy/Global Administrative state: 0" 6
    fi
    Print "   Policy Operational state: " 6
    if [ $(( $((0x$b21)) & 0x20)) = $((0x20)) ];then
      Print "      Policy is actively monitoring defined trigger (power or thermal) and will start enforcing the power limit if defined trigger is exceeded." 6
    else
      Print "      Policy is suspended so it cannot actively limit defined power limit. It may happen if one of the defined below events happens: " 6
      Print "        - suspend period is enforced" 6
      Print "        - there is a problem with trigger readings" 6
      Print "        - there is a host communication problem" 6
      Print "        - host is in Sx state" 6
      Print "        - host did not send End Of POST notification" 6
      Print "        - policy is administratively disabled " 6
    fi
    if [ $(( $((0x$b21)) & 0x40)) = $((0x40)) ];then
      Print "   Measurement state: Measurements in progress(host CPU is in S0 state and there are no problems with the readings reported to the remote console)" 6
    else
      Print "   Measurement state: Measurements are suspended(host CPU is in Sx state or in the S0 state with the readings problems reported to the remote console)" 6
    fi
    if [ $(( $((0x$b21)) & 0x80)) = $((0x80)) ];then
      Print "   Policy activation state : Policy is triggered and is actively limiting the defined power limit " 6
    else
      Print "   Policy activation state : Policy is not triggered" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMCapabilities
#
#   Description:
#  Function executes the  Get Node Manaher Capabilities command
#
#   Command code: C9h
#-----------------------------------------------------------------------------
GetNmCapabilities() {
  Print "Sending the Get Node Manager Capabilities command" 5 32
  Print " Get Node Manager Capabilities request:" 7 36
  read reqB4 reqB5 <<< $1
  Print "   DomainID: $reqB4" 7
  if [ $(( 0xF & $(($reqB5)) )) = 0 ];then
    Print "   Policy Trigger Type: No Policy Trigger" 7
  else
    Print "   Policy Trigger Type: Inlet Temperature Policy Trigger in [Celsius]" 7
  fi
  if [ $(( 0xF0 & $(($reqB5)) )) = $((0x10)) ];then
    Print "   Policy type: Power Control Policy" 7
  else
    Print "   Policy type: Unknown($(($reqB5)) [dec])" 7 31
  fi
  Print "" 7

  SendNmCmd "0xC9" "$1"
  read c2 c3 c4 c5 c6 c7 c8 c9 c10 c11 c12 c13 c14 c15 c16 c17 c18 c19 c20 c21 c22 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Capabilities response: " 6 36
    if [ "$c4$c3$c2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$c4$c$c2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$c4$c3$c2" 6
    fi
    Print "   Max Concurrent Settings: $(( 0x$c5 ))" 6  # Number of policies supported for the given policy trigger type and policy type
    Print "   Max Power/Thermal value to be settable as trigger or max Power Limit to be maintained  if Policy trigger type is equal to 0: $(( 0x$c7$c6 )) " 6
    Print "   Min Power/Thermal value to be settable as trigger or min Power Limit to be maintained  if Policy trigger type is equal to 0: $(( 0x$c9$c8 )) " 6
    Print "   Min Correction time: $(( 0x${c13}${c12}${c11}$c10 )) ms" 6
    Print "   Max Correction time: $(( 0x$c17$c16$c15$c14 )) ms" 6
    Print "   Min Statistics Reporting Period: $(( 0x$c19$c18 )) s" 6
    Print "   Max statistics reporting Period: $(( 0x$c21$c20 )) s" 6
    if [ $(( 0x$c22 & 0x7F )) = 0 ];then
      Print "   Domain limiting scope: platform power limiting " 6
    elif [ $((0x$c22 & 0x7F )) = 1 ];then
      Print "   Domain limiting scope: CPU power limiting " 6
    else
      Print "   Domain limiting scope: Unknown " 6 31
    fi
    if [ $(( 0x$c22 & 0x80 )) = 0 ];then
      Print "   Limitting based on: Wall input power. PSU input power " 6
    else
      Print "   Limitting based on: DC power. PSU output power or bladed system" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMVersion
#
#   Description:
#  Function executes the  Get Node Manager Version command
#
#   Command code: CAh
#-----------------------------------------------------------------------------
GetNMVersion() {
  Print "Sending the Get Node Manager Version command" 5 32
  SendNmCmd "0xCA" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Version response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print " " 5
    if [ $((0x$b5)) =  1 ];then
      Print "   Node Manager Version: supported Node Manager 1.0" 6
    elif [ $((0x$b5)) =  2 ];then
      Print "   Node Manager Version: supported Node Manager 1.5" 6
    elif [ $((0x$b5)) =  3 ];then
      Print "   Node Manager Version: supported Node Manager 2.0" 6
    else
      Print "   Node Manager Version: Unknown" 6 31
    fi
    Print " " 5
    if [ $((0x$b6)) = 1 ];then
      Print "   IPMI interface version: Node Manager IPMI version 1.0" 6
    elif [ $((0x$b6)) = 2 ];then
      Print "   IPMI interface version: Node Manager IPMI version 2.0" 6
    fi
    Print "   IPMI interface version: Patch version (binary encoded) 0x$b7" 6
    Print "   IPMI interface version: Major Firmware revision (binary encoded) 0x$b8" 6
    Print "   IPMI interface version: Minor Firmware revision (BCD encoded) 0x$b9" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetNMPowerDrawRange
#
#   Description:
#  Function executes the  Set Node Manager Power Draw Range command
#
#   Command code: CBh
#-----------------------------------------------------------------------------
SetNMPowerDrawRange(){
  Print "Sending the Set Node Manager Power Draw Range command" 5 32
  Print " Set Node Manager Power Draw Range request:" 7 36
  read reqB4 reqB5 reqB6 reqB7 reqB8 <<< $1
  Print "   DomainID: $(( $reqB4 & 0xF))" 7
  local reqMinPowerDraw=$(($reqB6*256+$reqB5))
  Print "   Minimum Power Draw in [Watts]: $reqMinPowerDraw" 7
  local reqMaxPowerDraw=$(($reqB8*256+$reqB7))
  Print "   Maximum Power Draw in [Watts]: $reqMaxPowerDraw" 7
  Print "" 7

  SendNmCmd "0xCB" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Node Manager Power Draw Range response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetNMAlertDestination
#
#   Description:
#  Function executes the  Set Node Manager Alert Destination command
#
#   Command code: CEh
#-----------------------------------------------------------------------------
SetNMAlertDestination(){
  Print "Sending the Set Node Manager Alert Destination command" 5 32
  Print " Set Node Manager Alert Destination request:" 7 36
  read reqB4 reqB5 reqB6 <<< $1
  Print "   BMC channel number over which alert from BMC to management console will be send: $(( $reqB4 & 0x0F ))" 7
  if [ $(( 0x80 & $(($reqB4)) )) = $((0x80)) ];then
    Print "   destination information operation: unregister alert receiver" 7
  else
    Print "   Destination information operation: register alert receiver " 7
  fi

  Print "   Destination Selector/Operation: $(($reqB5)) [dec]" 7

  Print "   Alert String Selector:" 7
  if [ $(( 0x7F & $(($reqB6)) )) = 0 ];then
    Print "     use volatile Alert String         " 7
  else
    Print "     non-volatile string selector($(( $((0x$b7 )) & 0x7F )) [dec])       " 7
  fi
  if [ $(( 0x80 & $(($reqB6)) )) = $((0x80)) ];then
    Print "     send Alert String identified by  following string selector" 7
  else
    Print "     don\'t send an Alert String " 7
  fi
  Print "" 7

  SendNmCmd "0xCE" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Node Manager Alert Destination response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMAlertDestination
#
#   Description:
#  Function executes the  Get Node Manager Alert Destination command
#
#   Command code: CFh
#-----------------------------------------------------------------------------
GetNMAlertDestination(){
  Print "Sending the Get Node Manager Alert Destination command" 5 32
  SendNmCmd "0xCF" "$1"
  read b2 b3 b4 b5 b6 b7 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Node Manager Alert Destination response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   BMC channel number over which alert from BMC to management console will be send: $(( $((0x$b5)) & 0x0F)) " 6
    if [ $(( $((0x$b5)) & 0x80)) = $((0x80)) ];then
      Print "   Destination information operation: Configuration invalid. Alert receiver not registered. Alerts are blocked." 6
    else
      Print "   Destination information operation: Configuration valid. Alert receiver registered" 6
    fi
    if [ $(( $((0x$b6)) & 0x0F  )) = 0  ];then
      Print "   Destination Selector/Operation: use volatile destination" 6
    else
      Print "   Destination Selector/Operation: $(( $((0x$b6)) & 0x0F  ))[dec]" 7
    fi
    Print "   Alert String Selector:" 6
    if [ $(( $((0x$b7 )) & 0x7F )) = 0 ];then
    Print "     use volatile Alert String" 6
    else
      Print "     non-volatile string selector($(( $((0x$b7 )) & 0x7F )) [dec])"  6
    fi
    if [ $(( $((0x$b7)) & 0x80 )) = 0 ];then
      Print "     don't send an Alert String" 6
    else
      Print "     send Alert String identified by following string selector" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetTotalPowerBudget
#
#   Description:
#  Function executes the  Set Total Power Budget command
#
#   Command code: D0h
#-----------------------------------------------------------------------------
SetTotalPowerBudget() {
  Print "Sending the Set Total Power Budget command" 5 32
  Print " Set Total Power Budget request:" 7 36
  read reqB4 reqB5 reqB6 <<< $1
  Print "   DomainID: $reqB4" 7
  local reqTargetPowerBudget=$(($reqB6*256+$reqB5))
  Print "   Target Power Budget: $reqTargetPowerBudget" 7
  Print "" 7

  SendNmCmd "0xD0" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Total Power Budget response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetTotalPowerBudget
#
#   Description:
#  Function executes the  Get Total Power Budget command
#
#   Command code: D1h
#-----------------------------------------------------------------------------
GetTotalPowerBudget() {
  Print "Sending the Get Total Power Budget command" 5 32
  Print " Get Total Power Budget request:" 7 36
  read reqB4 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "" 7

  SendNmCmd "0xD1" "$1"
  read b2 b3 b4 b5 b6 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Total Power Budget response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   Target power budget that should be maintained by the Power Budget Control Service: $((0x$b6$b5)) Watts" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetMaxAllowedCPUPStateTState
#
#   Description:
#  Function executes the  Set Max Allowed CPU P-State/T-State command
#
#   Command code: D2h
#-----------------------------------------------------------------------------
SetMaxAllowedCPUPStateTState() {
  Print "Sending the Set Max Allowed CPU P-State/T-State command" 5 32
  Print " Set Max Allowed CPU P-State/T-State request:" 7 36
  read reqB4 reqB5 reqB6 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "   P-State number to be set: $(($reqB5))" 7
  Print "   T-State number to be set: $(($reqB6))" 7
  Print "" 7

  SendNmCmd "0xD2" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Max Allowed CPU P-State/T-State response:" 6
    if [ "$b4$b$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetMaxAllowedCPUPStateTState
#
#   Description:
#  Function executes the  Get Max Allowed CPU P-State/T-State command
#
#   Command code: D3h
#-----------------------------------------------------------------------------
GetMaxAllowedCPUPStateTState() {
  Print "Sending the Get Max Allowed CPU P-State/T-State command" 5 32
  Print " Get Max Allowed CPU P-State/T-State request:" 7 36
  read reqB4 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "" 7

  SendNmCmd "0xD3" "$1"
  read b2 b3 b4 b5 b6 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Max Allowed CPU P-State/T-State response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   Current maximum P-State: $(( 0x$b5 ))" 6
    Print "   Current maximum T-State: $(( 0x$b6 ))" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNumberOfPStatesTStates
#
#   Description:
#  Function executes the  Get Number Of P-States/T-States command
#
#   Command code: D4h
#-----------------------------------------------------------------------------
GetNumberOfPStatesTStates() {
  Print "Sending the Get Number Of P-States/T-States command" 5 32
  Print " Get Number Of P-States/T-States request:" 7 36
  read reqB4 <<< $1
  Print "   DomainID: $reqB4" 7
  Print "" 7

  SendNmCmd "0xD4" "$1"
  read b2 b3 b4 b5 b6 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Number Of P-States/T-States response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print "   Number of P-States available on the platform: $(( 0x$b5 ))" 6
    Print "   Number of T-States available on the platform: $(( 0x$b6 ))" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SetHostCPUdata
#
#   Description:
#  Function executes the  Set Host CPU data command
#
#   Command code: D6h
#-----------------------------------------------------------------------------


SetHostCPUdata() {
  Print "Sending the Set Host CPU data command" 5 32
  Print " Set Host CPU data request:" 7 36
  read reqB4 reqB5 reqB6 reqB7 reqB8 reqB9 reqB10 reqB11 reqB12 reqB13 reqB14 reqB15 reqB16 reqB17 reqB18 reqB19 reqB20 reqB21 reqB22 reqB23 reqB24 reqB25 <<< $1
  Print "   DomainID: $(( $reqB4 & 0xF ))" 7
  if [ $(( $(($reqB5)) &  0x80 )) = $((0x80)) ]; then
    Print "   End of POST notification enabled" 7
  else
    Print "   End of POST notification disabled" 7
  fi
  if [ $(( $(($reqB5)) &  0x10 )) = $((0x10)) ]; then
    Print "   Host CPU discovery data is provided " 7
    local NumOfPStates=$(($reqB6))
    case $NumOfPStates in
      0)
        Print "   P-states are disabled by the user" 7
        ;;
      1)
        Print "   CPU does not support more P-states or in the multiprocessor environment some processors installed on board do not match the lowest number processor power-consumption parameters" 7
        ;;
      *)
        Print "   Actual number of supported P-States by the lowest number processor: $NumOfPStates" 7
        ;;
    esac
    local NumOfTStates=$(($reqB7))
    case $NumOfTStates in
      0)
        Print "   T-states are disabled by the user" 7
        ;;
      *)
        Print "   Actual number of supported T-States by the lowest number processor: $NumOfTStates" 7
        ;;  
    esac
    Print "   Number of installed CPUs/socket: $(($reqB8))" 7
    Print "   Processor Discovery Data for the lowest number processor: $reqB16 $reqB15 $reqB14 $reqB13 $reqB12 $reqB11 $reqB10 $reqB9" 7
    Print "   Processor Discovery Data 2 for the lowest number processor: $reqB24 $reqB23 $reqB22 $reqB21 $reqB20 $reqB19 $reqB18 $reqB17" 7
    Print "   Reading from PECI for the lowest number processor: $(($reqB25))" 7
  fi
  Print "" 7

  SendNmCmd "0xd6" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Set Host CPU data response:" 6 36
    if [ "$b4$b$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}


SetPSUConfiguration() {
  Print "Send Set PSU Configuration" 5 32
  Print " Set PSU Configuration request:" 7 36
  read reqB4 reqB5 reqB6 reqB7 reqB8 reqB9 reqB10 reqB11 reqB12 <<< $1
    Print "   DomainID: $reqB4" 7
    Print "" 7
    tab=( $reqB5 $reqB6 $reqB7 $reqB8 $reqB9 $reqB10 $reqB11 $reqB12 )
    index=0
    while [ $index -lt 8 ];do
    if [ $(( ${tab[$index]} & 0x01 )) = 1 ]; then
      Print "    PSU mode: the PSU is installed and lack of power readings should be reported to the Management Console" 6
    else
      Print "    PSU mode: the PSU is installed or may be attached in the future" 6
    fi
    Print "    PMBUS PSU address $(( $index + 1)): $(( $(( ${tab[$index]} & 0xFE)) >> 1)) [dec]" 6
    let "index = index + 1"
  done

  SendNmCmd "0xd7" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
    if [ "$error" -eq 0 ]; then
      Print "\n Set PSU Configuration response:" 6 36
    if [ "$b4$b$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

GetPSUConfiguration() {
  Print "Send Get PSU Configuration" 5 32
  Print " Get PSU Configuration request:" 7 36
  read reqB4 <<< $1
  Print "   DomainID: $(( $reqB4 & 0xF))" 7
  Print "" 7

  SendNmCmd "0xd8" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get PSU Configuration response:" 6 36
    if [ "$b4$b3$b2" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    Print " " 6
    Print "    DomainId: $(( 0x$b5 & 0x0f )) " 6
    Print " " 6
    tab=( $b6 $b7 $b8 $b9 $b10 $b11 $b12 $b13 )
    index=0
    while [ $index -lt 8 ]; do
      if [ $(( 0x${tab[$index]} & 0x01 )) = 1 ]; then
        Print "    PSU mode: the PSU is installed and lack of power readings should be reported to the Management Console" 6
      else
        Print "    PSU mode: the PSU is installed or may be attached in the future" 6
      fi
      bn=$(( 0x${tab[$index]} ))
      let " address = bn>>=1 "
      Print "    PMBUS PSU address $(( $index + 1 )): $address" 6
      let "index = index + 1"
      Print " " 6
    done
  fi
  Print " " 5
}

SendRAWPeci() {
  Print "Send: Send RAW Peci (Hex)" 5 32
  SendPeciCmd "0xdd" "$1"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
  Print " Send RAW Peci response:" 6
    Print "   Write FCS Value:  $b2" 6
    Print "   Read FCS Value:  $b3" 6
    Print "   PECI Response Data:  $b4 $b5 $b6 $b7 $b8 $b9 $b10 $b11" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetDeviceId
#
#   Description:
#  Function executes the Get Device Id command
#
#   Command code: 01h
#-----------------------------------------------------------------------------
GetDeviceId() {
  Print "Sending the Get Device ID command" 5 32

  SendIpmiCmd "0x06 0x01"
  read b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "\n Get Device ID response:" 6 36
    Print " Device ID: 0x$b2" 6
    Print " Device Revision: " 6
    Print "    Device Revision, binary encoded: $(( 0x$b3 & 0x0F ))" 6
    if [ $(( 0x$b3 & 0x80 )) -eq 0 ]; then
      Print "    device does not provide Device SDRs " 6
    fi
    Print " Firmware Revision 1: " 6
    if [ $(( 0x$b4 & 0x80 )) -eq 0 ]; then
      Print "    Device avaliable: normal operation " 6
    else
      Print "    Device avaliable: device firmware update or self-initialization in progress " 6
    fi
    Print "    Major Firmware Revision: $(( 0x$b4 & 0x7F ))" 6
    Print " Firmware Revision 2 " 6
    Print "    Minor Firmware Revision. BCD encoded: $(( (0x$b5 & 0xF0) >> 4 ))$(( 0x$b5 & 0x0F )) " 6
    Print " IPMI Version: $(( 0x$b6 & 0x0F )).$(( 0x$b6 & 0xF0 ))" 6
    Print " Additional Device Support: " 6
    if [ $(( 0x$b7 & 0x80 )) -eq 0 ]; then
      Print "    Not a chassis Device " 6
    fi
    if [ $(( 0x$b7 & 0x40 )) -eq 0 ]; then
      Print "    Not a Bridge " 6
    fi
    if [ $(( 0x$b7 & 0x20 )) -eq $(( 0x20 )) ]; then
      Print "    IPMB Event Generator " 6
    fi
    if [ $(( 0x$b7 & 0x10 )) -eq 0 ]; then
      Print "    Not a IPMB Event Receiver " 6
    fi
    if [ $(( 0x$b7 & 0x08 )) -eq 0 ]; then
      Print "    FRU Inventory Device " 6
    fi
    if [ $(( 0x$b7 & 0x04 )) -eq 0 ]; then
      Print "    SEL Device " 6
    fi
    if [ $(( 0x$b7 & 0x02 )) -eq 0 ]; then
      Print "    SDR Repository Device " 6
    fi
    if [ $(( 0x$b7 & 0x01 )) -eq 1 ]; then
      Print "    Sensor Device " 6
    fi

    if [ "$b10$b9$b8" = "000157" ]; then
      Print "   Manufacturers ID: 0x$b10$b9$b8 (Intel)" 6
    else
      Print "   Manufacturers ID: 0x$b10$b9$b8" 6
    fi
    Print " Product ID Minor Version: 0x$b11 " 6
    Print " Product ID Major Version: 0x$b12 " 6
    Print " Implemented version of SPS Firmware IPMI command specification BCD encoded: $(( (0x$b13 & 0xF0) >> 4 )).$(( 0x$b13 & 0x0F )) " 6

    Print " SPS Firmware build number BCD encoded: $(( (0x$b14 & 0xF0) >> 4 )).$(( 0x$b14 & 0x0F )) " 6
    Print " SPS Firmware last digit of build number and patch number BCD encoded: $(( (0x$b15 & 0xF0) >> 4)).$(( 0x$b15 & 0x0F )) " 6
    case $(( 0x$b16 & 0x03 )) in
      0)
        Print " Image flags: recovery image " 6
        ;;
      1)
        Print " Image flags: operational image 1 " 6
        ;;
      2)
        Print " Image flags: operational image 2 " 6
        ;;
    esac
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  SendRawPeci
#
#   Description:
#   Function executes the Send Raw Peci command
#
#   Command code: 40h
#-----------------------------------------------------------------------------
SendRawPeci() {
  Print "Send Send Raw Peci." 5 32
  SendNmCmd "0x40" "$1"
  read b2 b3 b4 ExtraByte <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    if [ "$b4$b3$b2" = "000157" ]; then
        Print "Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
      else
        Print "Manufacturers ID: 0x$b4$b3$b2" 6
    fi
    if [ "$ExtraByte" != "" ]; then
      Print " Send RAW Peci response: $ExtraByte" 6
    fi
  fi
  Print " " 5
}


#-----------------------------------------------------------------------------
#   Function:  NMPowerCharacterizationLaunchRequest
#
#   Description:
#   Function executes the NM Power Characterization launch request command
#
#   Command code: 60h
#-----------------------------------------------------------------------------
NMPowerCharacterizationLaunchRequest() {
  Print "Send NM Power Characterization launch request command." 5
  SendNmCmd "0x60" "$1"
  read b2 b3 b4 <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    if [ "$b4$b3$b2" = "000157" ]; then
        Print "Manufacturers ID: 0x$b4$b3$b2 (Intel)" 6
      else
        Print "Manufacturers ID: 0x$b4$b3$b2" 6
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetNMPowerCharacterizationRange
#
#   Description:
#   Function executes the Get Node Manager Power Characterization Range command
#
#   Command code: 61h
#-----------------------------------------------------------------------------
GetNMPowerCharacterizationRange() {
  Print "Send Get Node Manager Power Characterization Range command." 5
  SendNmCmd "0x61" "$1"
  read mId1 mId2 mId3 ts1 ts2 ts3 ts4 max1 max2 min1 min2 eff1 eff2 <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    if [ "$mId3$mId2$mId1" = "000157" ]; then
        Print "Manufacturers ID: 0x$mId3$mId2$mId1 (Intel)" 6
      else
        Print "Manufacturers ID: 0x$mId3$mId2$mId1" 6
    fi
    Print "Time Stamp: 0x$ts4$ts3$ts2$ts1" 6
    Print "Maximum Power Draw in [Watts]: 0x$max2$max1" 6
    Print "Minimum Power Draw in [Watts]: 0x$min2$min1" 6
    Print "Efficient Power Draw in [Watts]: 0x$eff2$eff1" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetSelTime
#
#   Description:
#   Function executes the Get Sel Time command
#
#   Command code: 48h
#-----------------------------------------------------------------------------
GetSelTime() {
  Print "Sending the Get Sel Time command" 5 32
  Print " Get Sel Time request:" 7 36
   SendIpmiCmd "0x0A 0x48"
   if [ $error != 0 ]; then
      Print "  Error sending Ipmi command: Get Sel Time. Response" 0 31
      return 1
   fi
   read b1 b2 b3 b4 b5  <<< $rsp
   GetSelTimestamp=0x$b4$b3$b2$b1
   Print "  Timestamp: $GetSelTimestamp" 6
   return 0
}

#-----------------------------------------------------------------------------
#   Function:  Get PCIe SMBus Slot Card Info"
#
#   Description:
#   Function executes the Get PCIe SMBus Slot Card Info command
#
#   Command code: e8h
#-----------------------------------------------------------------------------
GetPCIeSlotCardInfo() {
  Print "Get PCIe SMBus Slot Card Info"  7 36
   SendIpmiCmd "0x30 0xe8 $1"
   if [ $error != 0 ]; then
      Print "  Error sending Ipmi command: Get PCIe SMBus Slot Card Info. Response" 0 31
      return 1
   fi
   read TotNumOfCardsDetected CommProtSuppByProxy CommProtSuppByCardInstance AddProtBus SlotNumber IdSlaveAddres <<< $rsp

   Print " Total number of management-capable PCIe cards detected: $TotNumOfCardsDetected" 6
   Print " Command Protocols Supported by the Proxy: $CommProtSuppByProxy" 6
   Print " Command Protocols Supported by this Card instance: $CommProtSuppByCardInstance" 6
   Print " Address/Protocol/Bus: $AddProtBus" 6
   Print " Slot Number: $SlotNumber" 6
   Print " Identifier/Slave Address: $IdSlaveAddres" 6
   return 0
}

#-----------------------------------------------------------------------------
#   Function:  Slot IPMB"
#
#   Description:
#   Function executes the Slot IPMB command
#
#   Command code: 51h
#-----------------------------------------------------------------------------
SlotIPMB() {
  Print "Slot IPMB"  7 36
   SendIpmiCmd "0x3e 0x51 $1"
   if [ $error != 0 ]; then
      Print "  Error sending Ipmi command: Get PCIe SMBus Slot Card Info. Response" 0 31
      return 1
   fi
   Print "Response Data: $rsp"  6
   return 0
}

#-----------------------------------------------------------------------------
#   Function:  GetCUPSCapabilities
#
#   Description:
#   Function executes the Get CUPS Capabilities command
#
#   Command code: 64h
#-----------------------------------------------------------------------------
GetCUPSCapabilities() {
  Print "Send Get CUPS Capabilities command." 5
  SendNmCmd "0x64"
  read mId1 mId2 mId3 cupsCap cupsConf cupsVer rsv <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    if [ "$mId3$mId2$mId1" = "000157" ]; then
        Print "Manufacturers ID: 0x$mId3$mId2$mId1 (Intel)" 6
      else
        Print "Manufacturers ID: 0x$mId3$mId2$mId1" 6
    fi
    Print "CUPS  Capabilities: 0x$cupsCap" 6
    Print "CUPS policy configuration: 0x$cupsConf" 6
    Print "CUPS version: 0x$cupsVer" 6
    Print "Reserved: 0x$rsv" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetSensorReading
#
#   Description:
#   Function executes the Get Sensor Reading command
#
#   Command code: 2dh
#-----------------------------------------------------------------------------
GetSensorReading() {
  Print "Send Get Sensor Reading command." 5
  SendIpmiCmd "$SE_NET_FUN 0x2d $1"
  read sensRead sensReadSet sensReadB1 sensReadB2 <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    Print "Sensor reading: 0x$sensRead" 6
    Print "Settings: 0x$sensReadSet" 6
    Print "Byte 1: 0x$sensReadB1" 6
    if [ "$sensReadB2" != "" ]; then
      Print "Byte 2: 0x$sensReadB2" 6 # byte 2 is optional, available for discrete reading sensors only
    fi
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetCUPSData
#
#   Description:
#   Function executes the Get CUPS Data command
#
#   Command code: 65h
#-----------------------------------------------------------------------------
GetCUPSData() {
  Print "Send Get CUPS Data command." 5
  SendNmCmd "0x65" "$1"
  read mId1 mId2 mId3 cupsData <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    if [ "$mId3$mId2$mId1" = "000157" ]; then
        Print "Manufacturers ID: 0x$mId3$mId2$mId1 (Intel)" 6
      else
        Print "Manufacturers ID: 0x$mId3$mId2$mId1" 6
    fi
    Print "Requested parameter data: $cupsData" 6
  fi
  Print " " 5
}

#-----------------------------------------------------------------------------
#   Function:  GetDengateCap
#
#   Description:
#   Function executes the Get Dengate Capabilities command
#
#   Command code: b7h
#-----------------------------------------------------------------------------
GetDengateCap() {
  Print "Send Get Dengate Capabilities command." 5
  SendNmCmd "0xb7"
  read mId1 mId2 mId3 dengateData <<< $rsp
  if [ "$error" -eq 0 ]; then
    Print "Response received successfully." 5
    if [ "$mId3$mId2$mId1" = "000157" ]; then
        Print "Manufacturers ID: 0x$mId3$mId2$mId1 (Intel)" 6
      else
        Print "Manufacturers ID: 0x$mId3$mId2$mId1" 6
    fi
    Print "Requested parameter data: $dengateData" 6
  fi
  Print " " 5
}
