#!/bin/sh
# Copyright (c) 2014 Synology Inc. All rights reserved.

CURR_SCRIPT_NAME="$(basename $0)"

# DSM related
SYNO_SHARE_TOOL="/usr/syno/sbin/synoshare"
SERVICETOOL="/usr/syno/bin/servicetool"
SYNOPKG_TOOL="/usr/syno/bin/synopkg"
SYNO_SERVICE_TOOL="/usr/syno/sbin/synoservice"
SYNOINFO_CONF="/etc/synoinfo.conf"
DEF_SYNOINFO_CONF="/etc.defaults/synoinfo.conf"
MODEL_NAME="$(/bin/get_key_value "${DEF_SYNOINFO_CONF}" "upnpmodelname")"
DS_SERIAL_NUMBER="$(cat /proc/sys/kernel/syno_serial)"
SYNO_WEBAPI="/usr/syno/bin/synowebapi"
NORMAL_SHUTDOWN_FILE="/var/.NormalShutdown"

# SS tmp files
SS_PKG_STARTING_FILE="/tmp/sspkgstarting"
SS_NON_RECORDING_FILE="/tmp/ss_non_recording"
SS_LEAVING_NON_RECORDING_FILE="/tmp/ss_leaving_non_recording"

# SS related
SS_PACKAGE_DIR="/var/packages/SurveillanceStation"
SS_TARGET_DIR="${SS_PACKAGE_DIR}/target"
SS_PKG_INFO_FILE="${SS_PACKAGE_DIR}/INFO"
SS_PKG_ENABLED_FILE="${SS_PACKAGE_DIR}/enabled"
SS_SCRIPTS_DIR="${SS_TARGET_DIR}/scripts"
S82_SCRIPT="${SS_SCRIPTS_DIR}/S82surveillance.sh"
SS_UTILS_SCRIPT="${SS_SCRIPTS_DIR}/ss_utils.sh"
SS_LOG_FILE="/var/log/surveillance.log"
SSCORED_NAME="sscored"
SS_EMAP_DIR_NAME="@SSEmap"
SS_EMAP_DIR="${SS_TARGET_DIR}/${SS_EMAP_DIR_NAME}"
SS_USER_PHOTO_DIR_NAME="@SSUserPhoto"
SS_USER_PHOTO_DIR="${SS_TARGET_DIR}/${SS_USER_PHOTO_DIR_NAME}"
SS_DATA_DIR_NAME="@SSData"
SS_DATA_DIR="${SS_TARGET_DIR}/${SS_DATA_DIR_NAME}"
SS_AXISACSCTRL_DIR="${SS_TARGET_DIR}/axis_acsctrl"

# SS share folder related
SS_SHARE_FOLDER_LINK="/var/services/surveillance"
DEFAULT_SS_SHARE_NAME="surveillance"
CREATE_SHARE_RETRY_CNT=3
SS_SHARE_DS_SETTING_CONF="${SS_SHARE_FOLDER_LINK}/ds_settings.conf"
CONF_KEY_DS_SERIAL_NUMBER="ds_serial_number"

# SS settings.conf related
SS_SETTINGS_CONF="${SS_PACKAGE_DIR}/etc/settings.conf"
CONF_KEY_ACTIVE_VOLUME="active_volume"
CONF_KEY_SYS_DB_BKP_PATH="system_db_backup_path"
CONF_KEY_AXISACSCTRL_DB_BKP_PATH="axisacsctrl_db_backup_path"

# SS DB related
USE_SQLITE=true
PGSQL_DIR="/usr/syno/pgsql/bin"
PGSQL_DB_NAME="surveillance"
PGSQL_DB_USER="postgres"
PGSQL_BIN_PSQL="${PGSQL_DIR}/psql -U ${PGSQL_DB_USER} ${PGSQL_DB_NAME}"
PGSQL_BIN_CREATEDB="${PGSQL_DIR}/createdb -U ${PGSQL_DB_USER} ${PGSQL_DB_NAME}"
PGSQL_BIN_PG_DUMP="${PGSQL_DIR}/pg_dump --inserts -U ${PGSQL_DB_USER} ${PGSQL_DB_NAME}"
PGSQL_BIN_DROPDB="${PGSQL_DIR}/dropdb -U ${PGSQL_DB_USER} ${PGSQL_DB_NAME}"
SQLITE3="/usr/syno/bin/sqlite3"
SS_SYSTEM_DB_NAME="system.db"
SS_SYSTEM_DB="${SS_TARGET_DIR}/${SS_SYSTEM_DB_NAME}"
SS_AXISACSCTRL_DB_NAME="axisacsctrl.db"
SS_AXISACSCTRL_DB="${SS_TARGET_DIR}/${SS_AXISACSCTRL_DB_NAME}"
SS_AXISACSEVTLOG_DB_NAME="axisacsevtlog.db"
SS_AXISACSEVTLOG_DB="${SS_SHARE_FOLDER_LINK}/${SS_AXISACSEVTLOG_DB_NAME}"
SS_REC_DB_NAME="recording.db"
SS_REC_DB="${SS_SHARE_FOLDER_LINK}/${SS_REC_DB_NAME}"
SS_REC_CNT_DB_NAME="recording_cnt.db"
SS_REC_CNT_DB="${SS_SHARE_FOLDER_LINK}/${SS_REC_CNT_DB_NAME}"
SS_REC_SQL_SCRIPT="${SS_TARGET_DIR}/scripts/sql/recording.sql"
SS_REC_CNT_SQL_SCRIPT="${SS_TARGET_DIR}/scripts/sql/recording_cnt.sql"
SS_AXISACSEVTLOG_SQL_SCRIPT="${SS_AXISACSCTRL_DIR}/sql/axisacsevtlog.sql"
SS_SQL_UPGRADE_SCRIPT="${SS_SCRIPTS_DIR}/sql/sql.sh"
SS_AXISACSCTRL_SQL_UPGRADE_SCRIPT="${SS_AXISACSCTRL_DIR}/sql/sql.sh"
SS_SNAPSHOT_DB_NAME="snapshot.db"
SS_SNAPSHOT_DB="${SS_SHARE_FOLDER_LINK}/${SS_SNAPSHOT_DB_NAME}"
SS_SNAPSHOT_SQL_SCRIPT="${SS_TARGET_DIR}/scripts/sql/snapshot.sql"
SS_UPGRADER="${SS_TARGET_DIR}/bin/ssupgrader"
SS_SHAREFOLDERCHECKER="${SS_TARGET_DIR}/bin/sssharefolderchecker"

# Private functions

__IsSSEnabled()
{
	[ -f "${SS_PKG_ENABLED_FILE}" ] && echo true || echo false
}

__CreateSSShareFolder()
{
	local VolumePath="$1"

	env -i HTTP_X_SYNO_RELAY_LANGUAGE=enu ${SYNO_WEBAPI} --exec api=SYNO.Core.Share method=create \
		version=1 name='"'${DEFAULT_SS_SHARE_NAME}'"' \
		shareinfo='{"name":"'${DEFAULT_SS_SHARE_NAME}'","vol_path":"'${VolumePath}'","desc":"System default shared folder"}' \
		> /dev/null 2>&1
}

__SynoShareGetSSShareInfo()
{
	echo "$(${SYNO_SHARE_TOOL} --get "${DEFAULT_SS_SHARE_NAME}")"
}

__GetSSShareFolderVolPath()
{
	echo "$(__SynoShareGetSSShareInfo | grep "Path" | awk '{print $2}' | sed 's/^[\.]*//g' | \
			sed 's/^\[//g' | sed 's/\]$//g')"
}

__IsSSShareFolderLinkAlive()
{
	if [ -L "${SS_SHARE_FOLDER_LINK}" -a -d "$(readlink "${SS_SHARE_FOLDER_LINK}")" ]; then
		echo true
	else
		echo false
	fi
}

__CheckSSShareFolderLink()
{
	if [ false == $(__IsSSShareFolderLinkAlive) ]; then
		SSDebugLog "Service link missing. Call synoservice to re-link."
		${SYNO_SERVICE_TOOL} --setshare "${DEFAULT_SS_SHARE_NAME}" > /dev/null 2>&1
	fi
}

__CheckSSSharePrivilege()
{
	local ShareInfo="$(__SynoShareGetSSShareInfo)"
	local GotRWUsersPriv="$(echo "${ShareInfo}" | grep "RW list" | grep "@users")"
	local GotROAdminGrpPriv="$(echo "${ShareInfo}" | grep "RO list" | grep "@administrators")"

	if [ -n "${GotRWUsersPriv}" ]; then
		SSDebugLog "Surveillance shared folder rw list set to none."
		${SYNO_SHARE_TOOL} --setuser "${DEFAULT_SS_SHARE_NAME}" RW = '' > /dev/null 2>&1
	fi

	if [ -z "${GotROAdminGrpPriv}" ]; then
		SSDebugLog "Surveillance shared folder ro list set to admin,@administrators."
		${SYNO_SHARE_TOOL} --setuser "${DEFAULT_SS_SHARE_NAME}" RO = 'admin,@administrators' > /dev/null 2>&1
	fi
}

__PrepareLocalServiceShare()
{
	local VolumePath="$1"
	local CreateRetryCnt=0

	while [ -z "$(__GetSSShareFolderVolPath)" -a ${CREATE_SHARE_RETRY_CNT} -gt ${CreateRetryCnt} ]; do
		__CreateSSShareFolder "${VolumePath}"
		CreateRetryCnt=$((CreateRetryCnt+1))
		sleep 1
	done

	if [ -z "$(__GetSSShareFolderVolPath)" ]; then
		local AliveVolumePath="$(${SERVICETOOL} --get-alive-volume)"

		__CreateSSShareFolder "${AliveVolumePath}"
		if [ -z "$(__GetSSShareFolderVolPath)" ]; then
			SSDebugLog "Failed to create surveillance share folder."
			return 1
		fi
	fi

	__CheckSSShareFolderLink
	__CheckSSSharePrivilege

	return 0
}

__IsHARunning()
{
	local SupportHA=$(/bin/get_key_value "${DEF_SYNOINFO_CONF}" "support_ha")
	local RunHA=$(/bin/get_key_value "${SYNOINFO_CONF}" "runha")

	[ "yes" == "${SupportHA}" -a "yes" == "${RunHA}" ] && echo true || echo false
}

# Public functions

PackageInfoGet()
{
	if [ -f "${SS_PKG_INFO_FILE}" ]; then
		SURVEILLANCE_PKG_NAME=$(/bin/get_key_value "${SS_PKG_INFO_FILE}" package)
		SURVEILLANCE_PKG_VERSION=$(/bin/get_key_value "${SS_PKG_INFO_FILE}" version)
		SURVEILLANCE_PKG_VENDOR=$(/bin/get_key_value "${SS_PKG_INFO_FILE}" maintainer)
	else
		echo "Surveillance station package info file does not exist!";
		exit 1;
	fi
}

IsModelEDS14()
{
	if [ "EDS14" == "${MODEL_NAME}" ] ; then
		echo true
	else
		echo false
	fi
}

SurveillanceStationRestart()
{
	${S82_SCRIPT} restart
}

SSDebugLog()
{
	local CurrDate="$(date +'%b %e %H:%M:%S')"
	local LogMsg="$1"

	echo "${LogMsg}"
	echo "${CurrDate} ${CURR_SCRIPT_NAME}: ${LogMsg}" >> "${SS_LOG_FILE}"
}

IsNonRecordingMode()
{
	if [ -f ${SS_LEAVING_NON_RECORDING_FILE} ]; then
		echo false
	else
		[ -f ${SS_NON_RECORDING_FILE} ] && echo true || echo false
	fi
}

UpdateNonRecordingMode()
{
	if [ false == $(__IsSSShareFolderLinkAlive) ]; then
		SSDebugLog "Enter non-recording mode"
		SetNonRecordingMode true
	else
		SSDebugLog "Leave non-recording mode"
		SetNonRecordingMode false
	fi
}

# usage: SetNonRecordingMode (true|false) TRIGGER_SSCTL_REASON
SetNonRecordingMode()
{
	local blSet="$1"
	local TriggerSsctlReason="$2"
	local blTriggerSsctl=false

	if [ -n "${TriggerSsctlReason}" ]; then
		blTriggerSsctl=true
	fi

	if [ true == ${blSet} ]; then
		if [ true == ${blTriggerSsctl} ]; then
			${S82_SCRIPT} trigger-ssctl pre-enter-non-rec-mode "${TriggerSsctlReason}"
		fi

		touch ${SS_NON_RECORDING_FILE}

		if [ true == ${blTriggerSsctl} ]; then
			${S82_SCRIPT} trigger-ssctl post-enter-non-rec-mode "${TriggerSsctlReason}"
		fi
	else
		if [ true == ${blTriggerSsctl} ]; then
			${S82_SCRIPT} trigger-ssctl pre-leave-non-rec-mode "${TriggerSsctlReason}"
		fi

		touch ${SS_LEAVING_NON_RECORDING_FILE}

		#Update ds serial number in ds_setting.conf after pre-leave-non-rec-mode
		SetValueToConfFile "${SS_SHARE_DS_SETTING_CONF}" "${CONF_KEY_DS_SERIAL_NUMBER}" "${DS_SERIAL_NUMBER}"

		#TODO: Some functions of the ssupgrader may run with no effects in non-rec-mode. It's nice to fix it.
		${SS_SQL_UPGRADE_SCRIPT}
		${SS_UPGRADER} 0
		${SS_AXISACSCTRL_SQL_UPGRADE_SCRIPT}
		${SS_SHAREFOLDERCHECKER}

		rm -f ${SS_LEAVING_NON_RECORDING_FILE}
		rm -f ${SS_NON_RECORDING_FILE}

		if [ true == ${blTriggerSsctl} ]; then
			${S82_SCRIPT} trigger-ssctl post-leave-non-rec-mode "${TriggerSsctlReason}"
		fi
	fi
}

# usage: GetValueFromConfFile CONF_FILE KEY
GetValueFromConfFile()
{
	/bin/get_key_value "$1" "$2"
}

# usage: SetValueToConfFile CONF_FILE KEY VALUE
SetValueToConfFile()
{
	local ConfFile="$1"
	local Key="$2"
	local Value="$3"
	local KeyValPair=$(grep "^${Key}=" "${ConfFile}")

	if [ -n "${KeyValPair}" ]; then
		# escape \
		Value=$(echo ${Value} | sed 's/\//\\\//g')

		sed -i "s/^${Key}=.*/${Key}=\"${Value}\"/g" "${ConfFile}"
	else
		echo "${Key}=\"${Value}\"" >> "${ConfFile}"
	fi
}

# usage: RemoveKeyFromConfFile CONF_FILE KEY
RemoveKeyFromConfFile()
{
	local ConfFile="$1"
	local Key="$2"
	local KeyValPair=$(grep "^${Key}=" "${ConfFile}")

	if [ -n "${KeyValPair}" ]; then
		sed -i "s/^${Key}=.*//g" "${ConfFile}"
	fi
}

IsSSCoredRunning()
{
	[ -n "$(pidof ${SSCORED_NAME})" ] && echo true || echo false
}

PkgGetString()
{
	local File=$1
	local Sec=$2
	local Key=$3

	sed -n '/^\['$Sec'\]/,/^'$Key'/s/'$Key'.*=.*"\(.*\)"/\1/p' "$File"
}

IsValidSSVolumeLocation()
{
	local Location="$1"
	local blRet=false

	if [ true == $(IsModelEDS14) ]; then
		if [ "USB" == "${Location}" ]; then
			blRet=true
		fi
	else
		if [ "INTERNAL" == "${Location}" ]; then
			blRet=true
		fi
	fi

	echo "${blRet}"
}

UpdateActiveVolume()
{
	if [ true == $(__IsSSShareFolderLinkAlive) ]; then
		local ShareFolderPath="$(readlink "${SS_SHARE_FOLDER_LINK}")"
		local ActiveVolumePath="$(dirname "${ShareFolderPath}")"
		SetValueToConfFile "${SS_SETTINGS_CONF}" "${CONF_KEY_ACTIVE_VOLUME}" "${ActiveVolumePath}"
	fi
}

# usage: PrepareShareFolder PREPARE_RECORDING_DB VOLUME_PATH
# options:
#	PREPARE_RECORDING_DB: [true|false]
PrepareShareFolder()
{
	local blPrepareRecDB="$1"
	local VolumePath="$2"
	local Ret

	if [ -z "${VolumePath}" ]; then
		VolumePath=$(GetValueFromConfFile "${SS_SETTINGS_CONF}" "${CONF_KEY_ACTIVE_VOLUME}")
	fi

	__PrepareLocalServiceShare "${VolumePath}"
	Ret="$?"
	if [ 0 -ne "${Ret}" ]; then
		return "${Ret}"
	fi

	UpdateActiveVolume

	if [ true == "${blPrepareRecDB}" -a true == $(__IsSSShareFolderLinkAlive) ]; then
		if [ ! -f "${SS_REC_DB}" ]; then
			SSDebugLog "Prepare recording db."
			${SQLITE3} "${SS_REC_DB}" < "${SS_REC_SQL_SCRIPT}" > /dev/null 2>&1
		fi
		if [ ! -f "${SS_REC_CNT_DB}" ]; then
			SSDebugLog "Prepare recording count db."
			${SQLITE3} "${SS_REC_CNT_DB}" < "${SS_REC_CNT_SQL_SCRIPT}" > /dev/null 2>&1
		fi
		if [ ! -f "${SS_AXISACSEVTLOG_DB}" ]; then
			SSDebugLog "Prepare axisacsevtlog db."
			${SQLITE3} "${SS_AXISACSEVTLOG_DB}" < "${SS_AXISACSEVTLOG_SQL_SCRIPT}" > /dev/null 2>&1
		fi
		if [ ! -f "${SS_SNAPSHOT_DB}" ]; then
			SSDebugLog "Prepare snapshot db."
			${SQLITE3} "${SS_SNAPSHOT_DB}" < "${SS_SNAPSHOT_SQL_SCRIPT}" > /dev/null 2>&1
		fi
	fi
}

ShouldHandleVolumeEvent()
{
	if [ false == $(IsNonRecordingMode) -a \
		 false == $(__IsHARunning) -a \
		 ! -f "${NORMAL_SHUTDOWN_FILE}" ]; then
		echo true
	else
		echo false
	fi
}

USBVolumeMounted()
{
	${S82_SCRIPT} trigger-ssctl usb-volume-mounted "$1"
}

ShareSnapshotRestored()
{
	${S82_SCRIPT} trigger-ssctl share-snapshot-restored
}

PrepareBackupDB()
{
	local File="$1"
	local Key="$2"
	local Name="$3"
	local BackupPath=$(GetValueFromConfFile "${File}" "${Key}")

	if [ -f "${BackupPath}/${Name}" ]; then
		echo "${BackupPath}/${Name}"
		return 0
	fi

	if [ -f "${SS_SHARE_FOLDER_LINK}/${Name}" ]; then
		echo "${SS_SHARE_FOLDER_LINK}/${Name}"
		return 0
	fi

	return 1
}
