#!/bin/sh
# Copyright (c) 2014 Synology Inc. All rights reserved.

. /var/packages/SurveillanceStation/target/scripts/SurveillanceStationCommon.sh

UPGRADE_ERR_LOG="/var/log/ssupgradeerr.log"
TMP_UPGRADE_BIN_LIST="/tmp/ssupgradebinlist"
UPGRAGE_SCRIPT_DIR="/var/packages/SurveillanceStation/target/scripts/sql/upgrade"
UPGRADE_SCRIPT_LIST=`find ${UPGRAGE_SCRIPT_DIR} -name "*.sh" | sort`
UPGRAGE_BIN_DIR="/var/packages/SurveillanceStation/target/bin"

DB_INST_SYSTEM="system"
DB_INST_RECORDING="recording"
DB_INST_RECORDING_CNT="recording_cnt"
DB_INST_SNAPSHOT="snapshot"
DB_INST_AXISACSEVTLOG="axisacsevtlog"

##### Function for upgrade scripts

SSUpgradeErrLog()
{
	local CurrDate=$(date)

	echo "$1"
	echo "${CurrDate}: $1" >> ${UPGRADE_ERR_LOG}
}

GetSqliteDBPath()
{
	case $1 in
	${DB_INST_SYSTEM})
		echo ${SS_SYSTEM_DB}
		;;
	${DB_INST_RECORDING})
		echo ${SS_REC_DB}
		;;
	${DB_INST_SNAPSHOT})
		echo ${SS_SNAPSHOT_DB}
		;;
	esac
}

ExecuteSqlCmd()
{
	local DBInstance="$1"

	if [ ${USE_SQLITE} == false ]; then
		${PGSQL_BIN_PSQL} -c "$2"
	else
		local DBPath=$(GetSqliteDBPath ${DBInstance})

		${SQLITE3} ${DBPath} "$2"
	fi
}

ExecuteSqlScriptFile()
{
	local DBInstance="$1"
	local FilePath="${UPGRAGE_SCRIPT_DIR}/$2"

	if [ ${USE_SQLITE} == false ]; then
		${PGSQL_BIN_PSQL} < ${FilePath}
	else
		local DBPath=$(GetSqliteDBPath ${DBInstance})

		${SQLITE3} ${DBPath} < ${FilePath}
	fi
}

#	usage: TestTableColumnToRunScript -d DBInstance -t TableName -c ColumnName -[eE] -s SqlScriptFile -b UpgradeBin
#	options:
#		-t <name>		Table name to test
#		-c <name>		Column name to test
#		-s <name>		Sql script file to run if condition meets
#		-d <name>		Optional. DB Instance: system or recording. Default value is system.
#		-e				Optional. Sql script file will run when column does exist
#		-E				Optional. Sql script file will run when column does NOT exist. The default behavior.
#		-b <name>		Optional. Upgrade binary to run if condition meets
#	ret value:
#		0				Condition doest NOT meet
#		1				Condition meets
TestTableColumnToRunScript()
{
	local DBInstance="${DB_INST_SYSTEM}"
	local TableName
	local ColumnName
	local CondToRunScript=1
	local SqlScriptFile
	local UpgradeBin
	local Ret

	while getopts "d:t:c:eEs:b:" opt "$@"; do
		case "$opt" in
			d)
				DBInstance="$OPTARG"
			;;
			t)
				TableName="$OPTARG"
			;;
			c)
				ColumnName="$OPTARG"
			;;
			e)
				CondToRunScript=0
			;;
			E) 
				CondToRunScript=1
			;;
			s) 
				SqlScriptFile="$OPTARG"
			;;
			b) 
				UpgradeBin="$OPTARG"
			;;
		esac
	done
	shift $((OPTIND-1))

	if [ true == $(IsNonRecordingMode) ]; then
		if [ "${DB_INST_RECORDING}" == "${DBInstance}" ]; then
			echo "Pass recording.db[${TableName}][${ColumnName}] check when in non-recording mode."
			return 0
		elif [ "${DB_INST_RECORDING_CNT}" == "${DBInstance}" ]; then
			echo "Pass recording_cnt.db[${TableName}][${ColumnName}] check when in non-recording mode."
			return 0
		elif [ "${DB_INST_AXISACSEVTLOG}" == "${DBInstance}" ]; then
			echo "Pass axisacsevtlog.db[${TableName}][${ColumnName}] check when in non-recording mode."
			return 0
		elif [ "${DB_INST_SNAPSHOT}" == "${DBInstance}" ]; then
			echo "Pass snapshot.db[${TableName}][${ColumnName}] check when in non-recording mode."
			return 0
		fi
	fi

	echo "Test ${TableName} table to check if column (${ColumnName}) exists"
	ExecuteSqlCmd "${DBInstance}" "SELECT ${ColumnName} FROM ${TableName} LIMIT 1;"
	Ret=$?

	if [ ${Ret} = "${CondToRunScript}" ]; then
		echo "Execute ${SqlScriptFile}"
		ExecuteSqlScriptFile "${DBInstance}" "${SqlScriptFile}"
		Ret=$?

		if [ ${Ret} != 0 ]; then
			SSUpgradeErrLog "Failed to upgrade surveillance database (${TableName} ${ColumnName}), ret: ${Ret}"
			SSUpgradeErrLog "script: ${SqlScriptFile}"
		fi

		if [ -n "${UpgradeBin}" ]; then
			echo "${UPGRAGE_BIN_DIR}/${UpgradeBin}" >> ${TMP_UPGRADE_BIN_LIST}
		fi

		return 1
	fi

	return 0
}

##### Functions for main function

DelTmpUpgradeBinList()
{
	rm -f ${TMP_UPGRADE_BIN_LIST} > /dev/null 2>&1
}

RunUpgradeScript()
{
	for Script in $UPGRADE_SCRIPT_LIST;
	do
		echo "Run script: ${Script}"
		. ${Script}
	done
}

RunUpgradeBin()
{
	local BinaryRet=0;

	if [ -f ${TMP_UPGRADE_BIN_LIST} ]; then
		cat ${TMP_UPGRADE_BIN_LIST} | while read Line;
		do
			echo "Run upgrade bin: ${Line}"
			${Line} > /dev/null 2>&1
			BinaryRet=$?
			if [ $BinaryRet != 0 ]; then
				SSUpgradeErrLog "Failed to run ${Line}, ret: $BinaryRet"
			fi
		done

		DelTmpUpgradeBinList
	fi
}

main()
{
	DelTmpUpgradeBinList
	RunUpgradeScript
	RunUpgradeBin
}

main "$@"
