#!/bin/sh
# Copyright (c) 2014 Synology Inc. All rights reserved.

SURVEILLANCE_INC_SCRP=/var/packages/SurveillanceStation/target/scripts/SurveillanceStationCommon.sh
. ${SURVEILLANCE_INC_SCRP}

TABLE_LIST_IN_REC_DB="^event$ ^event_ds_.* ^analyticsevent$ ^log$ ^log_ds_.* ^edgevideoclip$ ^snapshot$ ^snapshot_ds_.* ^actrulehistory$ ^customaudiopattern"

ReplaceSqlSymbolForSqlite()
{
	local SrcSqlFile="$1"

	if [ ! -f "${SrcSqlFile}" ]; then
		SSDebugLog "Wrong sql script file: ${SrcSqlFile}."
		exit 1
	fi

	echo "BEGIN;"
	cat "${SrcSqlFile}" | \
		sed "s/camid integer NOT NULL/camid integer PRIMARY KEY/g" | \
		sed "s/cameraid integer NOT NULL/cameraid integer PRIMARY KEY/g" | \
		sed "s/id integer NOT NULL/id integer PRIMARY KEY AUTOINCREMENT/g" | \
		sed "s/serial unique/integer PRIMARY KEY AUTOINCREMENT/g" | \
		sed "s/false/0/g" | \
		sed "s/true/1/g" | \
		sed "s/::character varying//g" | \
		sed "s/::text//g"
	echo "COMMIT;"
}

AddBusyTimeoutForSqliteFile()
{
	local BusyTimeout="$1"
	local SrcSqlFile="$2"

	if [ ! -f "${SrcSqlFile}" ]; then
		SSDebugLog "Wrong sql script file: ${SrcSqlFile}."
		exit 1
	fi

	echo ".timeout " ${BusyTimeout}
	cat "${SrcSqlFile}"
}

# usage: DropSqliteDBUnnecessaryTables DBPath IsSystemDB
# options:
#	IsSystemDB: [true|false]
DropSqliteDBUnnecessaryTables()
{
	local DBFile="$1"
	local blSystemDB="$2"
	local TableList=$(${SQLITE3} "${DBFile}" "SELECT name FROM sqlite_master WHERE type='table';")

	for Table in ${TableList}; do
		local blDropTable
		if [ ${blSystemDB} == true ]; then
			blDropTable=false
		else
			blDropTable=true
		fi

		for TableInRecDB in ${TABLE_LIST_IN_REC_DB}; do
			local TableExists=$(echo ${Table} | grep -i "${TableInRecDB}")

			if [ -n "${TableExists}" ]; then
				if [ ${blSystemDB} == true ]; then
					blDropTable=true
				else
					blDropTable=false
				fi
				break
			fi
		done

		if [ ${blDropTable} == true ]; then
			${SQLITE3} "${DBFile}" "DROP TABLE ${Table};"
		fi
	done
}

Sqlite3Exec()
{
	local DbPath="$1"
	local SqlCmd="$2"

	/usr/syno/bin/sqlite3 -cmd ".timeout 300000" "${DbPath}" "${SqlCmd}"
}

DumpSqliteData()
{
	local DbPath="$1"
	local Tables=$(echo "$2" | sed 's/,/\\|/g')

	if [ ! -f "${DbPath}" ]; then
		SSDebugLog "Wrong db file: ${DbPath}."
		exit 1
	fi

	if [ -z "${Tables}" ]; then
		Sqlite3Exec "${DbPath}" ".dump" | grep -i '^INSERT INTO' | grep -vi '^INSERT INTO "sqlite_sequence"'
	else
		Sqlite3Exec "${DbPath}" ".dump" | grep -i '^INSERT INTO "\('"${Tables}"'\)"'
	fi
}

# Create dummy event file (if not exist) under @CMS/
# since file & its EA file should appear in pair
CreateSlaveDsDummyEvt()
{
	# e.g. @CMS/ds1/cam1/20150205PM/@eaDir/cam120150205-165640-1423126600.mp4
	local CmsDir=$SS_SHARE_FOLDER_LINK"/@CMS"
	local TmpDummyEvtPaths="/tmp/ss-dummy-evt-path"
	local FindMinDepth=5

	if ! [ -d $CmsDir ]; then
		return
	fi

	find "$CmsDir" -mindepth $FindMinDepth -type d | sed 's|@eaDir/||g' > $TmpDummyEvtPaths

	while read -r DummyEvt; do
		[ -f "$DummyEvt" ] || > "$DummyEvt"
	done < $TmpDummyEvtPaths

	rm $TmpDummyEvtPaths
}

Usage()
{
	cat << EOF
Usage: $(basename $0) [OPTION]
	--replace-sql-symbol-for-sqlite SQL_FILE RESULT_FILE
	--drop-sqlite-unnecessary-tables SQLITE_DB_FILE IS_SYSTEM_DB
		IS_SYSTEM_DB: [true|false]
	--dump-sqlite-data SQLITE_DB_FILE RESULT_FILE TABLES
		TABLES: [table,table,...]
	--create-slave-ds-dummy-events
EOF
}

main()
{
	case $1 in
	--replace-sql-symbol-for-sqlite)
		ReplaceSqlSymbolForSqlite "$2" > "$3"
		;;
	--add-busy-timeout-for-sqlite-file)
		AddBusyTimeoutForSqliteFile "$2" "$3" > "$4"
		;;
	--drop-sqlite-unnecessary-tables)
		DropSqliteDBUnnecessaryTables "$2" "$3"
		;;
	--dump-sqlite-data)
		DumpSqliteData "$2" "$4" >> "$3"
		;;
	--create-slave-ds-dummy-events)
		CreateSlaveDsDummyEvt
		;;
	*)
		Usage
		;;
	esac
}

main "$@"
