#!/bin/sh
# Copyright (c) 2016 Synology Inc. All rights reserved.

SVS_PKG_DIR="/var/packages/SurveillanceStation"
SVS_TARGET_DIR="${SVS_PKG_DIR}/target"
SVS_PKE_ENABLE="${SVS_PKG_DIR}/enabled"
SVS_PKG_INFO="${SVS_PKG_DIR}/INFO"
SVS_SHARE_DATA_PATH="/var/packages/SurveillanceStation/target/@surveillance"
DB_BACKUP_DIR="${SVS_SHARE_DATA_PATH}/@SSDBBackup"

SQL_SH="${SVS_TARGET_DIR}/scripts/sql/sql.sh"
UPGRADER_BIN="${SVS_TARGET_DIR}/bin/ssupgrader"
SYS_SQLITE3_BIN="/bin/sqlite3"

SYSTEM_DB_NAME="system.db"
SYSTEM_DB_PATH="${SVS_TARGET_DIR}/${SYSTEM_DB_NAME}"
REC_DB_NAME="recording.db"
REC_DB_PATH="${SVS_SHARE_DATA_PATH}/${REC_DB_NAME}"

SVS_PKG_VERSION="$(/bin/get_key_value "${SVS_PKG_INFO}" version | cut -d '-' -f 2)"
MIN_SQLITE3_VERSION=3234

CheckEnv()
{
	[ -f "${SVS_PKE_ENABLE}" ] && \
		echo "SurveillanceStation is still running!" && exit 0
	[ ! -d "${SVS_TARGET_DIR}" ] && \
		echo "No SurveillanceStation target dir!" && exit 0
	[ ! -d "${DB_BACKUP_DIR}" ] && \
		echo "No DB Backup dir!" && exit 0
	[ ! -f "${SQL_SH}" ] && \
		echo "No sql.sh!" && exit 0
	[ ! -f "${UPGRADER_BIN}" ] && \
		echo "No ssupgrader!" && exit 0
	[ ! -f "${SYS_SQLITE3_BIN}" ] && \
		echo "No sqlite3!" && exit 0
	[ ${MIN_SQLITE3_VERSION} -gt ${SVS_PKG_VERSION} ] && \
		echo "Sqlite3 unsupported version[${SVS_PKG_VERSION}]." && exit 0
}

PrintUsage()
{
	local DbList=$(/bin/ls -1A "${DB_BACKUP_DIR}")
	local SysList=$(echo "${DbList}" | grep "${SYSTEM_DB_NAME}")
	local RecList=$(echo "${DbList}" | grep "${REC_DB_NAME}")

	echo "Usage: `/bin/basename $0` [-s name|-r name]"
	echo "s:"
	echo "${SysList}"
	echo "r:"
	echo "${RecList}"
}

Sqlite3Exec()
{
	$SYS_SQLITE3_BIN -init /dev/null -cmd ".timeout 300000" "$@"
}

CheckDb()
{
	local File="$1"
	local Name="$2"
	local Path="${DB_BACKUP_DIR}/${File}"
	local Table="camera"

	if [ "${REC_DB_NAME}" == "${Name}" ]; then
		Table="event"
	fi

	if [ ! -f "${Path}" ]; then
		echo "[${File}] not exist."
		exit 0
	fi

	Sqlite3Exec "${Path}" "SELECT 1 FROM ${Table} LIMIT 0;" > /dev/null 2>&1
	if [ 0 -ne "$?" ]; then
		echo "[${File}] unable to SELECT ${Table} table, invalid ${Name}."
		exit 0
	fi
}

CheckArgs()
{
	local SysFile="$1"
	local RecFile="$2"

	if [ -z "${SysFile}" ] && [ -z "${RecFile}" ]; then
		PrintUsage
		exit 0
	fi

	if [ -n "${SysFile}" ]; then
		CheckDb "${SysFile}" "${SYSTEM_DB_NAME}"
	fi

	if [ -n "${RecFile}" ]; then
		CheckDb "${RecFile}" "${REC_DB_NAME}"
	fi
}

DbFileCopy()
{
	local SrcName="$1"
	local DbPath="$2"
	local DbName="$(/bin/basename ${DbPath})"
	local SrcPath="${DB_BACKUP_DIR}/${SrcName}"
	local DbBkpPath="${DbPath}.${SVS_PKG_VERSION}.org"

	if [ ! -f "${SrcPath}" ]; then
		return
	fi

	echo "Restore ${DbName} by [${SrcName}]"
	/bin/mv "${DbPath}" "${DbBkpPath}"
	/bin/cp "${SrcPath}" "${DbPath}"

	echo "Backup original ${DbName} to [${DB_BACKUP_DIR}/${DbName}.${SVS_PKG_VERSION}.org]"
	/bin/mv "${DbBkpPath}" "${DB_BACKUP_DIR}"
}

DoRestore()
{
	local SysFile="$1"
	local RecFile="$2"

	DbFileCopy "${SysFile}" "${SYSTEM_DB_PATH}"
	DbFileCopy "${RecFile}" "${REC_DB_PATH}"

	echo "Run sql.sh"
	${SQL_SH} > /dev/null 2>&1

	echo "Run ssupgrader 0"
	${UPGRADER_BIN} 0 > /dev/null 2>&1
}

main()
{
	local SysFile
	local RecFile
	local OPTARG
	local OPTIND

	CheckEnv

	while getopts "s:r:" opt "$@"; do
		case "$opt" in
			s)
				SysFile="$OPTARG"
			;;
			r)
				RecFile="$OPTARG"
			;;
		esac
	done
	shift $((OPTIND-1))

	CheckArgs "${SysFile}" "${RecFile}"
	DoRestore "${SysFile}" "${RecFile}"
}

main "$@"
