#!/bin/sh

. "$(dirname $0)/define.sh"

#---------------------------- Definitions -------------------------------------
readonly REDIRECT_OUTPUT="tee -a ${SZF_DEF_LOG_FILE}"
readonly SZ_SERVER_IP=$(ifconfig | grep -m 1 inet\ addr | cut -d':' -f 2 | cut -d' ' -f 1)
readonly SZ_WEBMAN="http://${SZ_SERVER_IP}:5000/webman"
readonly SZ_SS="${SZ_WEBMAN}/3rdparty/SurveillanceStation"
readonly SZ_LOGIN_CGI="${SZ_WEBMAN}/login.cgi"
readonly SZF_LOGIN_USER_FILE="/tmp/current.users"
readonly SZF_CGI_FEED="$(dirname $0)/cgifeedDSM5.sh"
readonly SZ_LOGIN_CGI_FOR_CGIFEED="http://${SZ_SERVER_IP}:5000/webapi/auth.cgi"
readonly SZ_SS_TEST_RESP_FOR_CGIFEED="/usr/syno/synoman/webman/3rdparty/SurveillanceStation/cgi/camera.cgi"
readonly MONITOR_INTEVAL_SEC=25
readonly CPU_SLEEP_CHECK_SEC=5
readonly MAX_LOG_FILE_CNT=100
readonly USE_CGIFEED=false

readonly SZ_SS_TEST_RESP_CGI="${SZ_SS}/cgi/camera.cgi"
readonly SZ_SS_TEST_RESP_PARAM="webapiAuth=yes&action=enum&statusOnly=true&UserId=1024"
readonly SZF_SS_TEST_RESP_FILE="/tmp/camera.cgi"

readonly SZ_SS_TEST_WEBAPI="http://${SZ_SERVER_IP}:5000/webapi/entry.cgi"
readonly SZ_SS_WEBAPI_PARAM_CAM="api=SYNO.SurveillanceStation.Camera&method=List&version=8&start=0&basic=true&streamInfo=true&privilege=false&camStm=1"
readonly SZF_SS_RESP_FILE_CAM="/tmp/camera.webapi"
readonly SZ_SS_WEBAPI_PARAM_EVT="api=SYNO.SurveillanceStation.Event&method=List&version=4&locked=0&evtSrcType=1&evtSrcId=0&blIncludeSnapshot=true&includeAllCam=true&offset=0&limit=80"
readonly SZF_SS_RESP_FILE_EVT="/tmp/event.webapi"
readonly SZ_SS_WEBAPI_PARAM_LOG="api=SYNO.SurveillanceStation.Log&method=List&version=1&start=0&limit=100&filterCamera=false&dsId=0&srcType=0&all=false"
readonly SZF_SS_RESP_FILE_LOG="/tmp/log.webapi"

#---------------------------- Global variables --------------------------------
report_memory_usage=true
report_fps_bps=true
logout_admin=true
max_memory_usage=0
curr_memory_usage=0
max_calc_cnt=480
calc_cnt=0
cgi_calc_cnt=0
cpu_usage_sum=0
cgi_resp_sum=0
cookie=""
cgi_user_name="${SZ_DEF_USER_NAME}"
cgi_user_pw="${SZ_DEF_USER_PASSWD}"
camera_webapi_resp_sum=0
camera_webapi_calc_cnt=0
event_webapi_resp_sum=0
event_webapi_calc_cnt=0
log_webapi_resp_sum=0
log_webapi_calc_cnt=0

#---------------------------- Functions ---------------------------------------
print_help()
{
	cat << EOF
Performance monitoring script.
Calculate CPU usage and CGI response time.

Admin will be automatically logged out.
Logs are saved as ${SZF_DEF_LOG_FILE}.
Please use eth0 to get correct cgi response calculation.

Usage: $(basename $0) [OPTION]
	-h	show this help and exit
	-M	don't report memory usage
	-E	don't report total fps and bps
	-A	don't logout admin
	-c	set max calculation count. -1 to infinite. (default: ${max_calc_cnt})
	-u	set cgi user name (default: ${cgi_user_name})
	-p	set cgi user password (default: ${cgi_user_pw})
EOF
}

print_output()
{
	echo "$@" | tee -a ${SZF_DEF_LOG_FILE}
}

update_memory_usage()
{
	local swap_total=$(cat /proc/meminfo | grep "SwapTotal" \
			 | sed -n 's/SwapTotal:[^0-9]*\([0-9]*\).*/\1/p')
	local swap_free=$(cat /proc/meminfo | grep "SwapFree" \
			| sed -n 's/SwapFree:[^0-9]*\([0-9]*\).*/\1/p')
	local memory_usage=$(cat /proc/meminfo | \
						 awk 'NR==1{ used = $2 } \
							  NR>=2 && NR<=4{ used -= $2 } \
							  { used += $2 } \
							  { used -= $2 } END{ print used }')
	memory_usage=$(echo $(($memory_usage + $swap_total - $swap_free)))
	curr_memory_usage=${memory_usage}
	if [[ ${memory_usage} -gt ${max_memory_usage} ]]; then
		max_memory_usage=${memory_usage}
	fi
}

cat_cpu_stat()
{
	echo $(cat /proc/stat | grep "cpu ")
}

calc_cpu_total()
{
	echo $(echo $@ | awk '{print $2+$3+$4+$5+$6+$7+$8+$9+$10}')
}

calc_cpu_used()
{
	echo $(echo $@ | awk '{print $2+$3+$4+$7+$8}')
}

calc_cpu_iowait()
{
	echo $(echo $@ | awk '{print $6}')
}

get_timer_msec()
{
	echo $(cat /proc/timer_list | awk '/now at/ {printf "%d", $3/1000000}')
}

login_and_get_cgi_cookie()
{
	local login_param="api=SYNO.API.Auth&method=login&version=3&account=${cgi_user_name}&passwd=${cgi_user_pw}&session=SurveillanceStation"

	if [[ true == ${USE_CGIFEED} ]]; then
		print_output "replace new web API"
		# ${SZF_CGI_FEED} ${SZ_LOGIN_CGI_FOR_CGIFEED} "${login_param}" > /dev/null 2>&1
	else
		if [[ -f ${SZF_LOGIN_FILE} ]]; then
			rm -f ${SZF_LOGIN_FILE}
		fi
		
		wget --output-document="${SZF_LOGIN_USER_FILE}" --post-data="${login_param}" \
			 "${SZ_LOGIN_CGI_FOR_CGIFEED}" > /dev/null 2>&1
		cookie=$(cat ${SZF_LOGIN_USER_FILE}  | grep "sid" | sed -n 's/.*sid":"\([^"]*\)".*/\1/p')
		
		if [[ -z "${cookie}" ]]; then
			print_output "Failed to retrieve cookie. Please check ss-viewer is added. Exit."
			echo "${SZ_SERVER_IP}"
			return 1
		fi
		cookie="id=${cookie}"
	fi

	return 0
}

get_cgi_resp_sec()
{
	local time1
	local time2
	local success
	local RespDoc=$1
	local PostData=$2
	local TestCgi=$3

	if [[ -f ${RespDoc} ]]; then
		rm -f ${RespDoc}
	fi

	time1=$(get_timer_msec)

	if [[ true == ${USE_CGIFEED} ]]; then
		print_output "replace new web API"
		# ${SZF_CGI_FEED} -u ${cgi_user_name} ${SZ_SS_TEST_RESP_FOR_CGIFEED} \
		#				"${SZ_SS_TEST_RESP_PARAM}" > ${RespDoc} 2>&1
	else
		wget --no-cookies --header="Cookie: ${cookie}" \
			 --output-document="${RespDoc}" \
			 --post-data="${PostData}" ${TestCgi} \
			 > /dev/null 2>&1
	fi

	time2=$(get_timer_msec)

	success=$(grep "\"success\":true" ${RespDoc})
	if [[ -n "${success}" ]]; then
		if [[ -n ${time2} ]] && [[ 0 -ne ${time2} ]] && [[ -n ${time1} ]] && [[ 0 -ne ${time1} ]]; then
			echo $(($time2 - $time1))
		else
			echo 0
		fi
	else
		echo 0
	fi
}

calc_cpu_usage_and_cgi_resp()
{
	local cpu_stat_1
	local cpu_stat_2
	local cpu_total
	local cpu_total_1
	local cpu_total_2
	local cpu_used
	local cpu_used_1
	local cpu_used_2
	local cpu_iowait_1
	local cpu_iowait_2
	local cpu_iowait_percent
	local cpu_usage_percent

	local cgi_resp_sec
	local camera_webapi_resp_sec
	local event_webapi_resp_sec
	local log_webapi_resp_sec

	# Evaluate
	cpu_stat_1=$(cat_cpu_stat)

	sleep ${CPU_SLEEP_CHECK_SEC}

	cpu_stat_2=$(cat_cpu_stat)
	if [[ true == ${report_memory_usage} ]]; then
		update_memory_usage
	fi

	# Calculate cpu usage
	calc_cnt=$(($calc_cnt + 1))

	cpu_total_1=$(calc_cpu_total ${cpu_stat_1})
	cpu_used_1=$(calc_cpu_used ${cpu_stat_1})
	cpu_iowait_1=$(calc_cpu_iowait ${cpu_stat_1})
	cpu_total_2=$(calc_cpu_total ${cpu_stat_2})
	cpu_used_2=$(calc_cpu_used ${cpu_stat_2})
	cpu_iowait_2=$(calc_cpu_iowait ${cpu_stat_2})

	cpu_total=$((${cpu_total_2} - ${cpu_total_1}))
	cpu_used=$((${cpu_used_2} - ${cpu_used_1}))
	cpu_iowait_percent=$(echo $((${cpu_iowait_2} - ${cpu_iowait_1})) ${cpu_total} | awk '{print $1*100/$2}')
	cpu_usage_percent=$(echo ${cpu_used} ${cpu_total} | awk '{print $1*100/$2}')
	cpu_usage_sum=$(echo ${cpu_usage_percent} ${cpu_usage_sum} | awk '{print $1+$2}')

	# Calculate cgi response
	cgi_resp_sec=$(get_cgi_resp_sec ${SZF_SS_TEST_RESP_FILE} ${SZ_SS_TEST_RESP_PARAM} ${SZ_SS_TEST_RESP_CGI})
	if [[ 0 -ge ${cgi_resp_sec} ]]; then
		print_output "Failed to send cgi request."
	else
		cgi_resp_sum=$((${cgi_resp_sec} + ${cgi_resp_sum}))
		cgi_calc_cnt=$(($cgi_calc_cnt + 1))
	fi

	# Calculate camera webapi response
	camera_webapi_resp_sec=$(get_cgi_resp_sec ${SZF_SS_RESP_FILE_CAM} ${SZ_SS_WEBAPI_PARAM_CAM} ${SZ_SS_TEST_WEBAPI})
	if [[ 0 -ge ${camera_webapi_resp_sec} ]]; then
		print_output "Failed to send camera webapi request."
	else
		camera_webapi_resp_sum=$((${camera_webapi_resp_sec} + ${camera_webapi_resp_sum}))
		camera_webapi_calc_cnt=$(($camera_webapi_calc_cnt + 1))
	fi

	# Calculate event webapi response
	event_webapi_resp_sec=$(get_cgi_resp_sec ${SZF_SS_RESP_FILE_EVT} ${SZ_SS_WEBAPI_PARAM_EVT} ${SZ_SS_TEST_WEBAPI})
	if [[ 0 -ge ${event_webapi_resp_sec} ]]; then
		print_output "Failed to send event webapi request."
	else
		event_webapi_resp_sum=$((${event_webapi_resp_sec} + ${event_webapi_resp_sum}))
		event_webapi_calc_cnt=$(($event_webapi_calc_cnt + 1))
	fi

	# Calculate log webapi response
	log_webapi_resp_sec=$(get_cgi_resp_sec ${SZF_SS_RESP_FILE_LOG} ${SZ_SS_WEBAPI_PARAM_LOG} ${SZ_SS_TEST_WEBAPI})
	if [[ 0 -ge ${log_webapi_resp_sec} ]]; then
		print_output "Failed to send log webapi request."
	else
		log_webapi_resp_sum=$((${log_webapi_resp_sec} + ${log_webapi_resp_sum}))
		log_webapi_calc_cnt=$(($log_webapi_calc_cnt + 1))
	fi


	# Print summary
	print_output "NO. ${calc_cnt}"
	print_output "Date:" $(date)
	print_output "CPU Avg:" $(echo ${cpu_usage_sum} ${calc_cnt} | awk '{print $1/$2}') "%"
	print_output "CGI Resp Avg:" \
				 $(awk -v sum=${cgi_resp_sum} -v cnt=${cgi_calc_cnt} 'BEGIN{print sum/cnt}') \
				 "ms"
	print_output "Camera Webapi Resp Avg:" \
				 $(awk -v sum=${camera_webapi_resp_sum} -v cnt=${camera_webapi_calc_cnt} 'BEGIN{print sum/cnt}') \
				 "ms"
	print_output "Event Webapi Resp Avg:" \
				 $(awk -v sum=${event_webapi_resp_sum} -v cnt=${event_webapi_calc_cnt} 'BEGIN{print sum/cnt}') \
				 "ms"
	print_output "Log Webapi Resp Avg:" \
				 $(awk -v sum=${log_webapi_resp_sum} -v cnt=${log_webapi_calc_cnt} 'BEGIN{print sum/cnt}') \
				 "ms"

	if [[ 0 -ne ${max_memory_usage} ]]; then
		print_output "Max Memory Usage:" "${max_memory_usage}" "KB"
		print_output "Current Memory Usage:" "${curr_memory_usage}" "KB"
		
		
		local swap_total=$(cat /proc/meminfo | grep "SwapTotal" \
				 | sed -n 's/SwapTotal:[^0-9]*\([0-9]*\).*/\1/p')
		local swap_free=$(cat /proc/meminfo | grep "SwapFree" \
				| sed -n 's/SwapFree:[^0-9]*\([0-9]*\).*/\1/p')
		print_output "SwapTotal: ${swap_total} KB"
		print_output "SwapFree: ${swap_free} KB"
	fi

	# Print current status
	print_output
	print_output "CPU:" ${cpu_usage_percent} "%"
	print_output "CPU IOwait:" ${cpu_iowait_percent} "%"
	print_output "Load Avg:" $(cat /proc/loadavg)

	print_output "${SZ_CGI_RESP}:" ${cgi_resp_sec} "ms"
	print_output "Camera Webapi" ${camera_webapi_resp_sec} "ms"
	print_output "Event Webapi" ${event_webapi_resp_sec} "ms"
	print_output "Log Webapi" ${log_webapi_resp_sec} "ms"

	if [[ true == ${report_fps_bps} ]]; then
		print_output
		/var/packages/SurveillanceStation/target/bin/sscamera -E | ${REDIRECT_OUTPUT}
	fi

	print_output "==================================="
	print_output

	return 0
}

rotate_log_file()
{
	if [[ -f "${SZF_DEF_LOG_FILE}.${MAX_LOG_FILE_CNT}" ]]; then
		rm -f "${SZF_DEF_LOG_FILE}.${MAX_LOG_FILE_CNT}" > /dev/null 2>&1
	fi

	for i in $(seq $((${MAX_LOG_FILE_CNT} - 1)) -1 0)
	do
		if [[ -f "${SZF_DEF_LOG_FILE}.${i}" ]]; then
			mv "${SZF_DEF_LOG_FILE}.${i}" "${SZF_DEF_LOG_FILE}.$(($i+1))" > /dev/null 2>&1
		fi
	done

	if [[ -f "${SZF_DEF_LOG_FILE}" ]]; then
		mv "${SZF_DEF_LOG_FILE}" "${SZF_DEF_LOG_FILE}.0" > /dev/null 2>&1
	fi

	echo > ${SZF_DEF_LOG_FILE}
}

main()
{
	while getopts "hMEAc:u:p:" opt; do
		case "$opt" in
			h)
				print_help
				exit 0
			;;
			M)
				report_memory_usage=false
			;;
			E)
				report_fps_bps=false
			;;
			A)
				logout_admin=false
			;;
			c)
				if [[ 0 -eq $OPTARG ]]; then
					echo "Count is set to 0. Abort."
					exit 0
				fi
				max_calc_cnt="$OPTARG"
			;;
			u)
				cgi_user_name="$OPTARG"
			;;
			p)
				cgi_user_pw="$OPTARG"
			;;
		esac
	done
	shift $((OPTIND-1))

	# Rotate old log
	rotate_log_file
	
	# Logout admin
	if [[ true == ${logout_admin} ]]; then
		sed -i "s/.*admin.*//g" ${SZF_LOGIN_USER_FILE}
	fi

	# Prepare cookie for cgi request
	login_and_get_cgi_cookie
	if [[ 0 -ne $? ]]; then
		exit 1
	fi

	# Main Loop
	print_output "Start Date:" $(date)
	if [[ 0 -lt ${max_calc_cnt} ]]; then
		print_output "Max Calculation Times: ${max_calc_cnt}"
	fi
	print_output

	while [[ 0 -gt ${max_calc_cnt} ]] || [[ ${calc_cnt} -lt ${max_calc_cnt} ]]
	do
		if [[ 1 -ne ${max_calc_cnt} ]]; then
			sleep ${MONITOR_INTEVAL_SEC}
		fi

		calc_cpu_usage_and_cgi_resp
		if [[ 0 -ne $? ]]; then
			break
		fi
	done
	exit 0
}

main "$@"

