BEGIN;

CREATE TYPE video_type AS ENUM ('movie', 'tvshow', 'tvshow_episode', 'home_video', 'tv_record');
CREATE TABLE mapper (
	id serial NOT NULL,
	type video_type,
	CONSTRAINT mapper_pkey PRIMARY KEY (id)
)
WITHOUT OIDS;

CREATE TABLE video_file (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	path varchar(4096) NOT NULL,
	filesize int8 NOT NULL DEFAULT 0,
	duration integer NOT NULL DEFAULT 0,
	container_type varchar(255),
	video_codec varchar(255),
	frame_bitrate integer,
	frame_rate_num integer,
	frame_rate_den integer,
	video_bitrate integer,
	video_profile integer,
	ff_video_profile integer DEFAULT -99 NOT NULL,
	video_level integer,
	resolutionX integer,
	resolutionY integer,
	display_x integer,
	display_y integer,
	rotation integer DEFAULT 0 NOT NULL,
	audio_codec varchar(255),
	audio_bitrate integer,
	frequency integer,
	channel integer,
	updated char(1),
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT video_file_pkey PRIMARY KEY (id),
	CONSTRAINT video_file_ukey UNIQUE (path),
	CONSTRAINT video_file_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE
)
WITHOUT OIDS;
CREATE INDEX video_file_path_idx ON video_file USING btree (path);
CREATE INDEX video_file_create_date_idx ON video_file USING btree (create_date);

CREATE TABLE summary (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	summary text NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT summary_pkey PRIMARY KEY (id),
	CONSTRAINT summary_umapper UNIQUE (mapper_id),
	CONSTRAINT summary_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE plus_info (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	plus_info text NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT plus_info_pkey PRIMARY KEY (id),
	CONSTRAINT plus_info_umapper UNIQUE (mapper_id),
	CONSTRAINT plus_info_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE poster (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	lo_oid oid NOT NULL,
	md5 text NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT poster_pkey PRIMARY KEY (id),
	CONSTRAINT poster_ukey UNIQUE (id, lo_oid),
	CONSTRAINT poster_umapper UNIQUE (mapper_id),
	CONSTRAINT poster_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE RULE propagate_deletes_to_lob AS
    ON DELETE TO poster
    DO ALSO SELECT lo_unlink(OLD.lo_oid) AS lo_unlink;

CREATE RULE propagate_update_to_lob AS
    ON UPDATE TO poster
    DO ALSO SELECT lo_unlink(OLD.lo_oid) AS lo_unlink;

CREATE TABLE writer (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	writer varchar(255) NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT writer_pkey PRIMARY KEY (id),
	CONSTRAINT writer_mapper_ukey UNIQUE (writer, mapper_id),
	CONSTRAINT writer_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX writer_idx ON writer USING btree (writer);

CREATE TABLE director (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	director varchar(255) NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT director_pkey PRIMARY KEY (id),
	CONSTRAINT director_mapper_ukey UNIQUE (director, mapper_id),
	CONSTRAINT director_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX director_idx ON director USING btree (director);

CREATE TABLE actor (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	actor varchar(255) NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT actor_pkey PRIMARY KEY (id),
	CONSTRAINT actor_mapper_ukey UNIQUE (actor, mapper_id),
	CONSTRAINT actor_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX actor_idx ON actor USING btree (actor);

CREATE TABLE gnere (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	gnere varchar(255) NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT gnere_pkey PRIMARY KEY (id),
	CONSTRAINT gnere_mapper_ukey UNIQUE (gnere, mapper_id),
	CONSTRAINT gnere_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX gnere_idx ON gnere USING btree (gnere);

CREATE TABLE library (
	id serial NOT NULL,
	title varchar(255) NOT NULL,
	type video_type NOT NULL,
	is_public boolean NOT NULL DEFAULT true,
	CONSTRAINT library_pkey PRIMARY KEY (id),
	CONSTRAINT library_ukey UNIQUE (title)
)
WITHOUT OIDS;

CREATE TYPE privilege_type AS ENUM ('none', 'read_only', 'read_write');
CREATE TABLE library_privilege (
	id serial NOT NULL,
	uid bigint NOT NULL,
	library_id int NOT NULL,
	type privilege_type NOT NULL,

	CONSTRAINT privilege_pkey PRIMARY KEY (id),
	CONSTRAINT privilege_ukey UNIQUE (uid, library_id),
	CONSTRAINT privilege_library_fkey FOREIGN KEY (library_id)
		REFERENCES library (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX library_privilege_idx ON library_privilege USING btree (uid);

CREATE TABLE collection (
	id serial NOT NULL,
	uid bigint NOT NULL,
	title varchar(255) NOT NULL,
	is_smart boolean NOT NULL DEFAULT false,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT collection_pkey PRIMARY KEY (id),
	CONSTRAINT collection_ukey UNIQUE (title, uid)
)
WITHOUT OIDS;
CREATE INDEX collection_title_idx ON collection USING btree (title);

CREATE TABLE collection_map (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	collection_id serial NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT collection_map_pkey PRIMARY KEY (id),
	CONSTRAINT collection_mapper_ukey UNIQUE (collection_id, mapper_id),
	CONSTRAINT collection_map_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT collection_map_collection_fkey FOREIGN KEY (collection_id)
		REFERENCES collection (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE movie
(
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	library_id int,
	title varchar(255) NOT NULL,
	sort_title varchar(255) NOT NULL,
	tag_line varchar(255) NOT NULL,
	year integer,
	originally_available date,
	sort_time date,
	certificate varchar(255) DEFAULT '' NOT NULL,
	isLock boolean,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT movie_pkey PRIMARY KEY (id),
	CONSTRAINT movie_ukey UNIQUE (library_id, title, year),
	CONSTRAINT movie_umapper UNIQUE (mapper_id),
	CONSTRAINT movie_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT movie_library_fkey FOREIGN KEY (library_id)
		REFERENCES library (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX movie_title_idx ON movie USING btree (title);

CREATE TABLE tvshow (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	library_id int,
	title varchar(255) NOT NULL,
	sort_title varchar(255) NOT NULL,
	year integer,
	originally_available date,
	sort_time date,
	isLock boolean,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT tvshow_pkey PRIMARY KEY (id),
	CONSTRAINT tvshow_ukey UNIQUE (library_id, title, year),
	CONSTRAINT tvshow_umapper UNIQUE (mapper_id),
	CONSTRAINT tvshow_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT tvshow_library_fkey FOREIGN KEY (library_id)
		REFERENCES library (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX tvshow_title_idx ON tvshow USING btree (title);

CREATE TABLE tvshow_episode (
	id serial NOT NULL,
	tvshow_id serial NOT NULL,
	library_id int,
	mapper_id serial NOT NULL,
	tag_line varchar(255) NOT NULL,
	season integer,
	episode integer,
	year integer,
	originally_available date,
	sort_time date,
	certificate varchar(255) DEFAULT '' NOT NULL,
	isLock boolean,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT tvshow_episode_pkey PRIMARY KEY (id),
	CONSTRAINT tvshow_episode_ukey UNIQUE (season, episode, tvshow_id),
	CONSTRAINT tvshow_episode_umapper UNIQUE (mapper_id),
	CONSTRAINT tvshow_episode_tvshow_fkey FOREIGN KEY (tvshow_id)
		REFERENCES tvshow (id) MATCH SIMPLE
		ON UPDATE CASCADE,
	CONSTRAINT tvshow_episode_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT tvshow_episode_library_fkey FOREIGN KEY (library_id)
		REFERENCES library (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE home_video (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	library_id int,
	title varchar(255) NOT NULL,
	sort_title varchar(255) NOT NULL,
	record_time timestamp,
	record_time_utc timestamp,
	certificate varchar(255) DEFAULT '' NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT home_video_pkey PRIMARY KEY (id),
	CONSTRAINT home_video_ukey UNIQUE(library_id, title, record_time_utc),
	CONSTRAINT home_video_umapper UNIQUE (mapper_id),
	CONSTRAINT home_video_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT home_video_library_fkey FOREIGN KEY (library_id)
		REFERENCES library (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX home_video_title_idx ON home_video USING btree (title);

CREATE TABLE tv_record (
	id serial NOT NULL,
	mapper_id serial NOT NULL,
	title varchar(255) NOT NULL,
	sort_title varchar(255) NOT NULL,
	channel_name varchar(255) NOT NULL,
	record_time timestamp,
	record_time_utc timestamp,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT tv_record_pkey PRIMARY KEY (id),
	CONSTRAINT tv_record_ukey UNIQUE(title, record_time_utc, channel_name),
	CONSTRAINT tv_record_umapper UNIQUE (mapper_id),
	CONSTRAINT tv_record_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;
CREATE INDEX tv_record_title_idx ON tv_record USING btree (title);

CREATE TABLE watch_status (
	id serial NOT NULL,
	uid bigint NOT NULL,
	video_file_id serial NOT NULL,
	mapper_id serial NOT NULL,
	position integer NOT NULL,
	create_date timestamp DEFAULT current_timestamp,
	modify_date timestamp DEFAULT current_timestamp,
	CONSTRAINT watch_status_pkey PRIMARY KEY (id),
	CONSTRAINT watch_status_ukey UNIQUE (uid, video_file_id, mapper_id),
	CONSTRAINT watch_status_mapper_fkey FOREIGN KEY (mapper_id)
		REFERENCES mapper (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT watch_status_video_file_fkey FOREIGN KEY (video_file_id)
		REFERENCES video_file (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE collection_sharing (
	id varchar(64) NOT NULL,
	collection_id integer NOT NULL,
	avail_date date NOT NULL DEFAULT '1970-01-01',
	exp_date date NOT NULL DEFAULT '1970-01-01',
	permanent boolean NOT NULL DEFAULT true,
	CONSTRAINT sharing_pkey PRIMARY KEY (id),
	CONSTRAINT sharing_ukey UNIQUE (collection_id),
	CONSTRAINT sharing_collection_id_fkey FOREIGN KEY (collection_id)
		REFERENCES collection (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE collection_smart (
	collection_id integer NOT NULL,
	type varchar(16) NOT NULL,
	has_default_library boolean NOT NULL DEFAULT true,
	filter text NOT NULL,

	CONSTRAINT smart_pkey PRIMARY KEY (collection_id),
	CONSTRAINT smart_collection_id_fkey FOREIGN KEY (collection_id)
		REFERENCES collection (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE TABLE collection_has_library (
	collection_id integer NOT NULL,
	library_id integer NOT NULL,

	CONSTRAINT has_library_ukey UNIQUE (collection_id, library_id),
	CONSTRAINT has_library_collection_id_fkey FOREIGN KEY (collection_id)
		REFERENCES collection (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE,
	CONSTRAINT has_library_library_id_fkey FOREIGN KEY (library_id)
		REFERENCES library (id) MATCH SIMPLE
		ON UPDATE CASCADE ON DELETE CASCADE
)
WITHOUT OIDS;

CREATE VIEW original_collection_video_list AS
	(SELECT 0 AS type, UPPER(sort_title) AS title, 0 AS season, 0 AS episode, mapper_id, sort_time FROM movie) UNION
	(SELECT 2 AS type, UPPER(sort_title) AS title, 0 AS season, 0 AS episode, mapper_id, record_time AS sort_time FROM home_video) UNION
	(SELECT 3 AS type, UPPER(sort_title) AS title, 0 AS season, 0 AS episode, mapper_id, record_time AS sort_time FROM tv_record) UNION
	(SELECT 1 AS type, tv.t AS title, season, episode, mapper_id, sort_time FROM tvshow_episode LEFT JOIN (SELECT id, UPPER(sort_title) AS t FROM tvshow) as tv ON tvshow_episode.tvshow_id = tv.id);

CREATE VIEW tvshow_episode_with_title AS
	SELECT e.id, e.tvshow_id, e.library_id, e.mapper_id, e.tag_line, e.season, e.episode,
	       e.year, e.originally_available, e.sort_time, e.islock, e.create_date, e.modify_date, e.certificate, tv.sort_title FROM tvshow_episode AS e
	LEFT JOIN (SELECT id, UPPER(sort_title) AS sort_title FROM tvshow) AS tv ON e.tvshow_id = tv.id;

CREATE VIEW watch_status_view AS
	SELECT video_file.id AS video_file_id, video_file.mapper_id, video_file.duration, watch_status.uid, CAST(watch_status.position AS float) AS position, watch_status.modify_date
	FROM video_file LEFT JOIN watch_status ON video_file.id = watch_status.video_file_id;

COMMIT;
