/***************************************************************************/
/*  Copyright 1993 National Instruments Corporation. All Rights Reserved   */
/***************************************************************************/

#include <nivxi.h>
#include <gpib.h>
#include <formatio.h>
#include "wt1395.h"

/*= Wavetek 1395 Arbitrary Waveform Generator =============================*/
/*  LabWindows Instrument Driver                                           */
/*  Original Release: August 25, 1993                                      */
/*  By: Judson Morrison, LWILDP, National Instruments, Austin, TX          */
/*      Phone (800) 433-3488  Fax (512) 794-5732                           */
/*  Originally written in C                                                */
/*  Modification History:                                                  */
/*             August 94 - Restructured the function tree and added        */
/*                         functions to define the trace mode and define   */
/*                         trace data points using a single integer array. */
/*                         Also added the low level functions to read and  */
/*                         write directly to the instrument.  Previous     */
/*                         release of this driver was a beta version and   */
/*                         the appropriate corrections and alterations     */
/*                         were made to ensure proper execution.           */
/*                         Changed the response parameter of the WScmd     */
/*                         serial poll to a local unsigned short variable  */
/*                         that is cast to the character value returned by */
/*                         the poll.                                       */
/*                         Glen Wienecke, LWILDP, National Instruments     */
/*=========================================================================*/
/* Modifications:          Modified for MDLL Compatibility                 */
/*                                                                         */
/* Modification Date:      June 28, 1996                                   */
/*                                                                         */
/* Modified by:            DS, National Instruments, Austin, Texas         */
/*                         Phone (800) 433-3488  Fax (512) 794-5678        */
/*                                                                         */                                                                          
/***************************************************************************/ 

/*= INSTRUMENT TABLE ======================================================*/
/* log_addr array: contains the logical addresses of opened instruments.   */
/* bd array: contains the device descriptors returned by OpenDev (NI-GPIB only) */
/* interface array: specifies the type of interface (NI-VXI or NI-GPIB).   */
/* base_addr array: contains the base address in A16 space for the instr.  */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* wtk1395_err: the error variable for the instrument module                */
/*=========================================================================*/
static int log_addr[wtk1395_MAX_INSTR + 1];
static int bd[wtk1395_MAX_INSTR + 1];
static int interface[wtk1395_MAX_INSTR + 1];
static long base_addr[wtk1395_MAX_INSTR + 1];
static long A24_base_addr[wtk1395_MAX_INSTR + 1];
static int instr_cnt;
static int wtk1395_err;

/*= STATIC DATA ===========================================================*/
/* cmd : a buffer for GPIB and VXI I/O strings.                            */
/* timeout: a look up table used to associate GPIB timeout codes with Word */
/*          Serial timout codes.                                           */
/* bytes: contains the actual number of bytes transferred by WSrd, WSwrt,  */
/*        WSrdf, & WSwrtf.                                                 */
/*=========================================================================*/
static char cmd[9000];
static long timeout[18];
static long bytes;
static char buffer[32767];

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
static char *adv[2];
static char *dir[2];
static char *sweep[6];
static char *space[2];
static char *func[11];
static char *source[13];
static char *adv_source[4];
static char *refosc[4];
static char *trac_mode[2];
static char *boolean[2];
static char *modulation[2];
static char *sync[4];
static char *qual[5];
static char *clk[3];
static char *sync_marker[2];
/*=========================================================================*/

/*= UTILITY ROUTINES ======================================================*/
int wtk1395_open_instr (int, int, int *);
int wtk1395_close_instr (int);
int wtk1395_init_gpib (int);
int wtk1395_invalid_integer_range (int, int, int, int);
int wtk1395_invalid_longint_range (long, long, long, int);
int wtk1395_invalid_real_range (double, double, double, int);
int wtk1395_device_closed (int);
int wtk1395_read_data (int, char *, int);
int wtk1395_write_data (int, char *, int);
int wtk1395_read_data_file (int, char *);
int wtk1395_write_data_file (int, char *);
int wtk1395_read_reg(int, int, int *);
int wtk1395_set_timeout (int, int);
int wtk1395_poll (int, char *);
int wtk1395_get_status_reg (int, int *);
int wtk1395_write_data_no_end_sgnl(int, char *, int);
int wtk1395_read_err(int, int *);
int wtk1395_check_instr_status(int);
void wtk1395_setup_arrays (void);

/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries for ID, and       */
/*           initializes the instrument to a known state.                  */
/*=========================================================================*/
int wtk1395_init (int laddr, int platform, int id_query, int rest, int *instrID)
{
    int id, value;
    char response;

    if (wtk1395_invalid_integer_range (laddr, 0, 255, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (platform, 1, 2, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (id_query, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (rest, 0, 1, -4) != 0)
        return wtk1395_err;

    /* Initialize entry in Instrument Table and interface for instrument. */
    if (wtk1395_open_instr (laddr, platform, &id) != 0)
        return wtk1395_err;

/* MESSAGE BASED ID Query */
    if (id_query) {
        if (wtk1395_write_data (id, "*IDN?", 5) != 0)
            return wtk1395_err;

    /* Serial Polling for Message Available Bit */
        response = 0;
        while (response & 0x10 != 0x10) {
            if (wtk1395_poll (id, &response) != 0)
                return wtk1395_err;
        }
        if (wtk1395_read_data (id, cmd, 50) != 0)
            return wtk1395_err;

        Scan (cmd, "Wavetek Instruments, 1395");
        if (NumFmtdBytes () != 25)  {
            wtk1395_err = 223;
            return wtk1395_err;
        }
    }
    /* Initialize the instrument to a known state. */
    if (rest) {
        if (wtk1395_write_data (id, "*RST;*CLS", 9) != 0) {
            wtk1395_close_instr (id);
            return wtk1395_err;
        }
    }
    /* Initialize Instrument-Dependent Command arrays. */
    wtk1395_setup_arrays ();

    /* Find the base A24 address assigned to this instrument and put into */
    /* A24_base_addr array, use instrID as Array Index.                   */
    if (wtk1395_read_reg(id, 0x06, &value) != 0)
        return wtk1395_err;
    A24_base_addr[id] = (long)value * 0x100L;
    /* Set timeout to 10 seconds */
    wtk1395_set_timeout (id, 13);

    *instrID = id;
    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Continuous Wave                                               */
/* Purpose: This function configures the source frequency mode for         */
/*          continuous wave operation and sets the frequency value.        */
/*          This mode is used to output standard functions at a single     */
/*          frequency.                                                     */
/*=========================================================================*/
int wtk1395_cw (int instrID, double freq)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (freq, 1E-3, 50E6, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:MODE?");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 9000) != 0)
        return wtk1395_err;

    Scan (cmd, "%s>CW");
    if (NumFmtdBytes() != 2) {
        wtk1395_err = 10;
        return wtk1395_err;
    }

    Fmt (cmd, "%s<SOUR:FREQ:MODE CW;CW %f", freq);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Raster                                                        */
/* Purpose: This function configures the source frequency mode for         */
/*          continuous wave operation and sets the raster frequency        */
/*          value.                                                         */
/*=========================================================================*/
int wtk1395_raster (int instrID, double freq)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (freq, 1E-3, 50E6, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:MODE?");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 9000) != 0)
        return wtk1395_err;

    Scan (cmd, "%s>RAST");
    if (NumFmtdBytes() != 4) {
        wtk1395_err = 10;
        return wtk1395_err;
    }

    Fmt (cmd, "%s<SOUR:FREQ:MODE CW;RAST %f", freq);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Sweep                                                         */
/* Purpose: This function configures the instrument for the sweep mode,    */
/*          disables either the list or continuous wave output modes,      */
/*          sets start and stop frequencies, sets sweep triggering,        */
/*          selects sweep mode, direction, and scale.                      */
/*=========================================================================*/
int wtk1395_sweep (int instrID, double count, int direction, int mod, double time,
                   int spacing, double start, double stop)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (count, 1e0, 1e6, -2) !=0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (direction, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 5, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (time, 30.0E-3, 1.0E3, -5) !=0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (spacing, 0, 1, -6) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (start, 30E-3, 50E6, -7) !=0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (stop, 30E-3, 50E6, -8) !=0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<FREQ:MODE SWE;STAR %f;STOP %f;:SWE:COUN %f;DIR %s;MODE %s;TIME %f;SPAC %s",
        start, stop, count, dir[direction], sweep[mod], time, space[spacing]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: List                                                          */
/* Purpose: This function configures the instrument for the list mode,     */
/*          and programs the frequencies to be output by the instrument.   */
/*=========================================================================*/
int wtk1395_list (int instrID, double freq_list[], int length)
{
    int i, bytes;
    bytes = 0;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (length, 1, 1024, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (buffer, "%s<FREQ:MODE LIST;:LIST:POIN %d[b4]", length);
    bytes = NumFmtdBytes();
    for (i = 0; i < length; i++) {
        Fmt (buffer, "%s[a]<;FREQ %f,%d[b4]", freq_list[i], i);
        bytes += NumFmtdBytes();
    }
    if (wtk1395_write_data (instrID, buffer, bytes) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Standard                                                      */
/* Purpose:  This function configures the source function mode to SHAPE    */
/*           and selects one of the standard waveforms as the source       */
/*           function. This function also configures the trace mode for    */
/*           continuous wave operation, enables the elliptic output        */
/*           filter and enables the sync marker output to trigger on       */
/*           zero-crossings.                                               */
/*=========================================================================*/
int wtk1395_stand (int instrID, int type)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 10, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:FUNC:MODE FIX;SHAP %s", func[type] );
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Trace                                                         */
/* Purpose: This function configures the source function mode to USER      */
/*          and selects one of the trace waveforms as the source function. */
/*=========================================================================*/
int wtk1395_trac (int instrID, char trac[])
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:FUNC:MODE FIX;SHAP USER;USER %s", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Get A24 Shared Memory Data                                    */
/* Purpose: This function configures the source function mode to SHAPE and */
/*          uses data from the first 8 KBytes of A24 Shared Memory as      */
/*          the source function. This function only selects the trace in   */
/*          A24 memory as the current output function and does not save    */
/*          trace in memory (the function Write A24 to Trace Memory        */
/*          saves the trace to memory but does not select it as the output */
/*          function.)                                                     */
/*=========================================================================*/
int wtk1395_getdat (int instrID)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:FUNC:MODE FIX;SHAP SMEM");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Sequence                                                      */
/* Purpose: This function configures the source function mode to USER      */
/*          and selects sequence currently defined as the source for       */
/*          waveform generation.                                           */
/*=========================================================================*/
int wtk1395_sequence (int instrID)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:FUNC:MODE SEQ");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Trigger Enable                                                */
/* Purpose:  This function selects triggered or continous mode of          */
/*           operation. Trigger mode must be selected before the start,    */
/*           advance, or software trigger functions are valid.             */
/*=========================================================================*/
int wtk1395_trigen (int instrID, int state)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (state != 0)
        Fmt (cmd, "%s<INIT:CONT OFF");
    else
        Fmt (cmd, "%s<INIT:CONT ON");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Start Trigger                                                 */
/* Purpose:  This function selects triggered mode of operation, configures */
/*           the start trigger then waits for the specified trigger signal */
/*           to enable the current output function.                        */
/*=========================================================================*/
int wtk1395_starttrig (int instrID, int type, int polarity, int gate, long count, int mod)
{
    int response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 11, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (polarity, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (gate, 0, 1, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_longint_range (count, 1L, 1048575L, -5) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 1, -6) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRIG:SOUR:STAR %s;:TRIG:COUN %d[b4];GATE:STAT %s;MODE %s",
         source[type], count, boolean[gate], sync[mod]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Advance Trigger                                               */
/* Purpose:  This function selects triggered mode of operation and         */
/*           configures the advance trigger. The advance trigger is used   */
/*           to sequence through elements in a sequence table. A start     */
/*           trigger must be used to begin execution of the sequence table */
/*=========================================================================*/
int wtk1395_adtrig (int instrID, int type, int polarity, int gate, long count, int mod)
{
    int response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 3, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (polarity, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (gate, 0, 1, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_longint_range (count, 1L, 1048575L, -5) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 1, -6) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRIG:SOUR:ADV %s;:TRIG:COUN %d[b4];GATE:STAT %s;MODE %s",
         adv_source[type], count, boolean[gate], sync[mod]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Software Trigger                                              */
/* Purpose:  This function initiates a trigger signal (either advance or   */
/*           start) from the user's application if the trigger mode is     */
/*           enabled.                                                      */
/*=========================================================================*/
int wtk1395_softtrig (int instrID)
{
    int response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt(cmd, "%s<*TRG");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/*  Function:   Set Internal Trigger Rate                                  */
/*  Purpose:    This function sets the period of the trigger clock.        */
/*=========================================================================*/
int wtk1395_settrig (int instrID, double rate)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (rate, 1E-3, 1E4, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRIG:TIM %f", rate);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

    return wtk1395_err;
}
/* ========================================================================= */
/*  Function: Configure Markers                                              */
/*  Purpose:  This function configures the marker output of the instrument.  */
/* ========================================================================= */
int wtk1395_markers (int instrID, char name[], int sync, int type, int trig, int pos,
                     int points, int arr[])
{
    int bytes, i;

    bytes = 0;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR,  -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (sync, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 1, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (trig, 0, 1, -5) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (pos, 0, 1, -6) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID) != 0)
        return wtk1395_err;

 /* Send the ALL OFF command if position markers are to be turned off */
    if (pos == 0) {
        Fmt (cmd, "%s<MARK:SYNC:SOUR %s;:MARK:SYNC %d[b4];:MARK:TRIG:STAT %s,%d[b4];:MARK:POS:AOFF %s",
             sync_marker[type], sync, name, trig, name);
        if (wtk1395_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return wtk1395_err;
    }
 /* Turn on the desired marker positions */
    else {
        Fmt (buffer, "%s<MARK:SYNC:SOUR %s;:MARK:SYNC %d[b4];:MARK:TRIG:STAT %s,%d[b4];:MARK:POS:",
             sync_marker[type], sync, name, trig);
        bytes = NumFmtdBytes();
        for (i = 0; i < points; i++) {
            Fmt (buffer, "%s[a]<POIN %s,%d[b4];", name, arr[i]);
            bytes += NumFmtdBytes();
        }
        if (wtk1395_write_data (instrID, buffer, (bytes - 1)) != 0)
            return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function:  Set DC Offset Voltage                                        */
/* Purpose:   This function sets the DC offset voltage at the main output. */
/*=========================================================================*/
int wtk1395_offset (int instrID, double volts)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (volts, -7.5, 7.5, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:VOLT:LEV:OFFS %f", volts);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function:  Set Maximum Amplitude                                        */
/* Purpose:   This function sets the maximum amplitude for optimum         */
/*            output resolution of the output signal.                                           */
/*=========================================================================*/
int wtk1395_ampl (int instrID, double max)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (max, 0.0, 7.5, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:VOLT:LEV:IMM:AMPL %f", max);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Phase Lock                                                    */
/* Purpose:  This function synchronizes phase between two 1395's.          */
/*=========================================================================*/
int wtk1395_lock (int instrID, int state)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:PHAS:LOCK %s", boolean[state]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Phase Adjust                                                  */
/* Purpose:  This function configures a phase lag or lead for the output   */
/*           signal up to 180 degrees.                                     */
/*=========================================================================*/
int wtk1395_adjust (int instrID, int phase)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (phase, -180, 180, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:PHAS:ADJ %d[b4]", phase);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Output Enable                                                 */
/* Purpose:  This function enables the main output connector. This function*/
/*           must be called to receive an output.                          */
/*=========================================================================*/
int wtk1395_outen (int instrID, int state)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (state == 1)
        Fmt (cmd, "%s<OUTP:STAT ON");
    else
        Fmt (cmd, "%s<OUTP:STAT OFF");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Filter                                                        */
/* Purpose:  This function enables/disables the output filter and selects  */
/*           the type of filter either Bessel or Elliptic.                 */
/*=========================================================================*/
int wtk1395_filter (int instrID, int state, int type)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<OUTP:FILT:STAT %s", boolean[state]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

    if (type != 0)
        Fmt (cmd, "%s<OUTP:FILT:SEL ELL");
    else
        Fmt (cmd, "%s<OUTP:FILT:SEL BESS");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Sumbus Output                                                 */
/* Purpose:  This function configures the instrument to source sumbus      */
/*           signals from the backplane.                                   */
/*=========================================================================*/
int wtk1395_sumout (int instrID, int state)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 360, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<OUTP:SUMB:STAT %s", boolean[state]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Output Trigger                                                */
/* Purpose:  This function configures the source and qualifiers for the    */
/*           output trigger module.                                        */
/*=========================================================================*/
int wtk1395_outtrig (int instrID, int type, int line, int qualifier, int state)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (line, 0, 7, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (qualifier, 0, 4, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 1, -5) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<OUTP:TRIG:SOUR %s", qual[qualifier]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

    if (type != 0)
        Fmt (cmd, "%s<OUTP:ECLT%d[b4] %s", line, boolean[state]);
    else
        Fmt (cmd, "%s<OUTP:TTLT%d[b4] %s", line, boolean[state]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Output Clock                                                  */
/* Purpose:  This function configures the output module to source a clock  */
/*           from the CLK IN/OUT BNC and disables all other output modes.  */
/*=========================================================================*/
int wtk1395_clock (int instrID, int config, int mod, double freq)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (config, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 2, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_real_range (freq, 1E-1, 1E8, -4) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (config == 0) {
        switch (mod) {
            case 0: Fmt (cmd, "%s<CLOC:CONF OUTP;:OUTP:CLOC:FREQ %f", freq);
                    break;
            case 1: Fmt (cmd, "%s<CLOC:CONF OUTP;:OUTP:CLOC:SOUR RAST");
                    break;
            case 2: Fmt (cmd, "%s<CLOC:CONF OUTP;:OUTP:CLOC:SOUR SYNT");
                    break;
        }
    }
    else {
        Fmt (cmd, "%s<CLOC:CONF INP");
    }

    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Sumbus Input                                                  */
/* Purpose:  This function configures the instrument to receive sumbus     */
/*           inputs from the backplane and adjusts the attenution for the  */
/*           input signals.                                                */
/*=========================================================================*/
int wtk1395_sumin (int instrID, int state, int atten)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 360, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (atten, 0, 360, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:SUMB:STAT %s;ATT %d[b4]", boolean[state], atten);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Modulation                                                    */
/* Purpose:  This function configures the AM IN BNC to receive a           */
/*           modulating signal and selects a modulation mode.              */
/*=========================================================================*/
int wtk1395_mod (int instrID, int state, int mod)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (state, 0, 360, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:AM:STAT %s;MODE %s", boolean[state], modulation[mod]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Reference Oscillator                                          */
/* Purpose:  This function configures the source of the refernece          */
/*           oscillator.                                                   */
/*=========================================================================*/
int wtk1395_refosc (int instrID, int type)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (type, 0, 3, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SOUR:ROSC:SOUR %s", refosc[type]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Trace Playback Mode                                           */
/* Purpose:  This function configures the trace playback mode of the       */
/*           instrument.                                                   */
/*=========================================================================*/
int wtk1395_trace_mode (int instrID, int mod)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:MODE %s", trac_mode[mod]);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Create Trace                                                  */
/* Purpose:  This function reserves a name and location in trace memory    */
/*           for a function waveform.  This function must be performed     */
/*           before standard or user-defined traces can be copied into the */
/*           memory referenced by the new trace name.                      */
/*=========================================================================*/
int wtk1395_create (int instrID, char name[], int points)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (points, 1, 32678, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:DEF %s, %d[b4]", name, points);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Copy Trace                                                    */
/* Purpose:  This function copies a trace or a standard function into      */
/*           another location in trace memory referenced by the specified  */
/*           trace name.                                                   */
/*=========================================================================*/
int wtk1395_copy (int instrID, char sour[], char target[])
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:DATA %s, %s", target, sour);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Delete Trace                                                  */
/* Purpose:  This function deletes a trace in trac ememory.                */
/*=========================================================================*/
int wtk1395_delete (int instrID, int func, char name[])
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (func, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (func !=1)
        Fmt (cmd, "%s<TRAC:DEL:NAME %s", name);
    else
        Fmt (cmd, "%s<TRAC:DEL:ALL");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: List Trace Directory                                          */
/* Purpose:  This function returns a string with the names, sizes, start,  */
/*           and stop indices for each trace. The start and stop indices   */
/*           are determined by the trace limits.                           */
/*=========================================================================*/
int wtk1395_tradir (int instrID, int func, char dir[])
{
    int length = 0;
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (func, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

 /* Initialize the dir string */
    length = StringLength (dir);
    FillBytes (dir, 0, length, 0);

   if (func != 1)
        Fmt (cmd, "%s<TRAC:DIR?");
   else
        Fmt (cmd, "%s<TRAC:CAT?");
   if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
   }
   if (wtk1395_read_data (instrID, cmd, 9000) != 0)
        return wtk1395_err;

    if (Scan (cmd, "%s>%s", dir) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

   return wtk1395_err;
}

/*=========================================================================*/
/* Function: Available Trace Memory                                        */
/* Purpose:  This function returns a string with the number of points      */
/*           available for use and the number of point in use in trace     */
/*           memory.                                                       */
/*=========================================================================*/
int wtk1395_free (int instrID, long *mem, long *used)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:FREE?");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 100) != 0)
        return wtk1395_err;

    if (Scan (cmd, "%s>%d[b4],%d[b4]", mem, used) != 2) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Write Trace to File                                           */
/* Purpose:  This function returns an array of data points from a single   */
/*           trace in trace memory into a binary file.                     */
/*=========================================================================*/
int wtk1395_writefile (int instrID, char trac[], char file[])
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:DATA? %s", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data_file (instrID, file) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Write Waveform File to Instrument                             */
/* Purpose:  This function reads the data from a waveform file on disk,    */
/*           defines a waveform in the instrument which will hold the      */
/*           waveform and then uploads it to the instrument.               */
/*=========================================================================*/
int wtk1395_file_to_trace (int instrID, char file_name[], char wvfm_name[])
{
    int exists, byte_cnt, num_pts, hdl;
    long mem, used, num_bytes;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    /* Read the waveform file to see how many bytes are in the waveform */
    hdl = OpenFile (file_name, 1, 0, 0);
    if (hdl > 0) {
        byte_cnt = ReadFile (hdl, cmd, 15);
        if (byte_cnt <= 0) {
            wtk1395_err = 320;
            CloseFile (hdl);
            return wtk1395_err;
        }
    }
    CloseFile (hdl);
    if (Scan (cmd, "%s>#%d[b4w1]", &byte_cnt) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }
    if (Scan (cmd, "%s[i2]>%d[b4w*]", byte_cnt, &num_pts) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }
    num_pts /= 2;

    /* Check to see if there is enough free memory in the intrument to hold
       the waveform. */
    if (wtk1395_free (instrID, &mem, &used) != 0)
        return wtk1395_err;
    if (mem < num_pts) {
        wtk1395_err = 305;
        return wtk1395_err;
    }

    /* Define the trace and write the data to the instrument */
    Fmt (cmd, "%s<TRAC:DEF %s,%d[b4];:TRAC:DATA %s,", wvfm_name, num_pts, wvfm_name);
    if (wtk1395_write_data_no_end_sgnl (instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

    if (wtk1395_write_data_file (instrID, file_name) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: VXI Write to A24 Shared Memory                                */
/* Purpose:  This function writes trace data from a binary file (created   */
/*           with the Write Trace to File function) into A24 Shared Memory */
/*           which can then be stored in trace memory or used as the       */
/*           output function. This function is only valid for VXI-MXI or   */
/*           Embedded VXI operation.                                       */
/*=========================================================================*/
int wtk1395_writeA24 (int instrID, char file[])
{
    int ret, srcparms, destparms, file_handle, index, ref, word, word1, word2;
    long numbytes, destaddr;
    char *trac;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR,  -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID) != 0)
        return wtk1395_err;

    srcparms = 0x80;
    /*   Local, Non Privileged, Intel   */
    destparms = 0x82;
    /*   A24, Non Privileged, Intel   */

    if (GetFileInfo (file, &numbytes) != 1) {
        wtk1395_err = 102;
        return wtk1395_err;
    }
    file_handle = OpenFile (file, 1, 2, 0);
        if (file_handle < 0) {
            wtk1395_err = 103;
            return wtk1395_err;
        }
    /* Skips the Definite Length Arbitrary Block header */
    ret = SetFilePtr (file_handle, 6L, 0);

    if (ScanFile (file_handle, "%*i[zb2o01]>%*i[zb2]", (int)(numbytes), (int)(numbytes), buffer) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }
    if (ret < 0)  {
        CloseFile (file_handle);
        wtk1395_err = 100 + ret;
        return wtk1395_err;
    }
    else {
        CloseFile (file_handle);
    }
    if (interface[instrID] == NI_GPIB)
        wtk1395_err = 105;
    else {
        if (GetDevInfoLong (log_addr[instrID], 12, &destaddr) != 0) {
            wtk1395_err = 101;
            return wtk1395_err;
        }
        /* Moves the entire block of trace data */
        if (VXImove (srcparms, (long) buffer, destparms, destaddr, ((long)numbytes / 2), 2) != 0) {
            wtk1395_err = 235;
            return wtk1395_err;
        }
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: VXI Write A24 to Trace Memory                                 */
/* Purpose:  This function reads an array of trace data points from A24    */
/*           Shared Memory and writes into trace memory. If a trace with   */
/*           the specified name is not present, the function creates it.   */
/*           This function is only valid for VXI-MXI or Embedded VXI       */
/*           operation.                                                    */
/*=========================================================================*/
int wtk1395_writetrace (int instrID, char trac[], int exist, int size)
{
    int response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (exist, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (size, 1, 32767, -4) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (exist == 0) {
        Fmt (cmd, "%s<TRAC:DEF %s, %d[b4]", trac, size);
        if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
            return wtk1395_err;
    }
    Fmt (cmd, "%s<TRAC:DATA %s, SMEM", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Draw Line                                                     */
/* Purpose:  This function draws a line between two specified points in    */
/*           a trace. All points between the specified endpoints are       */
/*           over-written.                                                 */
/*=========================================================================*/
int wtk1395_line (int instrID, char trac[], int index1, int value1, int index2, int value2)
{
    int start, stop;
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (index1, 0, 32767, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (value1, 0, 4095, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (index2, 0, 32767, -5) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (value2, 0, 4095, -6) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

 /* Checking size of trac to range verify index values */
    Fmt (cmd, "%s<TRAC:LIM? %s", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 1000) != 0)
        return wtk1395_err;

    if (Scan (cmd, "%s>%d[b4],%d[b4]", &start, &stop) != 2) {
        wtk1395_err = 236;
        return wtk1395_err;
    }
    if ((index1 < start) || (index1 > stop)) {
        wtk1395_err = -3;
        return wtk1395_err;
    }
    if ((index2 < start) || (index2 > stop)) {
        wtk1395_err = -5;
        return wtk1395_err;
    }

    Fmt (cmd, "%s<TRAC:DATA:LINE %s,%i[b4],%i[b4],%i[b4],%i[b4]", trac, index1, value1, index2, value2);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Define Points                                                 */
/* Purpose:  This function defines the point values for a trace in memory. */
/*=========================================================================*/
int wtk1395_define_pts (int instrID, char trac[], int num_pts, short arr[])
{
    int byte_cnt;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

 /* Determine the number of digits in byte count for data */
    Fmt (cmd, "%s<%d[b4]", 2 * num_pts);
    byte_cnt = NumFmtdBytes ();

 /* Upload the waveform  */
    Fmt (buffer, "%s<TRAC:DATA %s,#%d[b4]%d[b4]", trac, byte_cnt, 2 * num_pts);
    byte_cnt = NumFmtdBytes ();
    Fmt (buffer, "%*d[zi*b2]<%*d[b2o10]", num_pts, byte_cnt, num_pts, arr);
    byte_cnt += NumFmtdBytes ();
    if (wtk1395_write_data (instrID, buffer, byte_cnt) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Change Point                                                  */
/* Purpose:  This function changes the value of a single point in a        */
/*           specified trace.                                              */
/*=========================================================================*/
int wtk1395_point (int instrID, char trac[], int index, int value)
{
    int start, stop;
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (value, 0, 4095, -4) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

 /* Checking size of trac to range verify index values */
    Fmt (cmd, "%s<TRAC:LIM? %s", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 1000) != 0)
        return wtk1395_err;

    if (Scan (cmd, "%s>%d[b4],%d[b4]", &start, &stop) != 2) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

    if ((index < start) || (index > stop)) {
        wtk1395_err = -3;
        return wtk1395_err;
    }

    Fmt (cmd, "%s<TRAC:DATA:POIN %s, %d[b4], %d[b4]", trac, index, value);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Re-Scale Trace                                                */
/* Purpose:  This function horizontally scales a trace to fit into a new   */
/*           memory location.                                              */
/*=========================================================================*/
int wtk1395_rescale (int instrID, char trac[], int points)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (points, 1, 32767, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:POIN %s, %d[b4]", trac, points);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Get Trace Limits                                              */
/* Purpose:  This function returns the playback limits of the specified    */
/*           trace.                                                        */
/*=========================================================================*/
int wtk1395_getlim (int instrID, char trac[], int *start, int *stop)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:LIM? %s", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
        }
    if (wtk1395_read_data (instrID, cmd, 1000) != 0)
        return wtk1395_err;
    if (Scan (cmd, "%s>%d[b4],%d[b4]", start, stop) != 2) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Get Point Value                                               */
/* Purpose:  This function returns the value of a trace data point         */
/*           specified by a trace name and point index.                    */
/*=========================================================================*/
int wtk1395_getpoint (int instrID, char trac[], int index, int *val)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (index, 0, 32767, -3) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:DATA:POIN? %s, %d[b4]", trac, index);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
        }
    if (wtk1395_read_data (instrID, cmd, 1000) != 0)
        return wtk1395_err;
    if (Scan (cmd, "%s>%d[b4]", val) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Get Trace Size                                                */
/* Purpose:  This function returns the number of points in a specified     */
/*           trace.                                                        */
/*=========================================================================*/
int wtk1395_getsize (int instrID, char trac[], int *val)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<TRAC:POIN? %s", trac);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
        }
    if (wtk1395_read_data (instrID, cmd, 1000) != 0)
        return wtk1395_err;
    if (Scan (cmd, "%s>%d[b4]", val) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Define Sequence                                               */
/* Purpose: This function configures the instrument for the sequence mode  */
/*          and sets the number of items in the sequence table. For the    */
/*          triggered mode of operation, edge or level triggering is       */
/*          selected. Synchronous or asynchronous selects whether the      */
/*          next sequence element is output after the completion of the    */
/*          current element or if the next element interrupts the current. */
/*=========================================================================*/
int wtk1395_defseq (int instrID, int indices, int sense, int mod)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (indices, 2, 4, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (sense, 0, 1, -3) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (mod, 0, 1, -4) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (sense != 0)
        Fmt (cmd, "%s<SEQ:LENG %d[b4];TRIG:SENS EDGE;MODE %s", indices,
             sync[mod]);
    else
        Fmt (cmd, "%s<SEQ:LENG %d[b4];TRIG:SENS LEV;MODE %s", indices,
             sync[mod]);

    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Link Sequence                                                 */
/* Purpose: This function creates and orders elements in the sequence      */
/*          table. The element index ranges from 0 to 3 (maximum of four   */
/*          elements) and sequences the functions in the seauence table.   */
/*          The trace name selects the element function, dwell specifies   */
/*          the number of times to cycle through each element, and advance */
/*          selects either triggered or automatic advancing to the next    */
/*          element.                                                       */
/*=========================================================================*/
int wtk1395_linkseq (int instrID, int index, char trac[], long dwell, int advance)
{
    int response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (index, 0, 3, -2) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_longint_range (dwell, 0L, 65535L, -4) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (advance, 0, 1, -5) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<SEQ:FUNC %s,%d[b4];DWEL %d[b4],%d[b4];ADV %s,%d[b4]",
         trac, index, dwell, index, adv[advance], index);
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*==========================================================================*/
/* Function:    Read Error Queue                                            */
/* Description: This function reads all the errors from the error queue and */
/*              returns the last error code.                                */
/*==========================================================================*/
int wtk1395_inst_err (int instrID, int * value)
{
    int test, err_code;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    err_code = 0;
    test = 1;
    while (test != 0) {
        if (wtk1395_read_err (instrID, &test) != 0)
            return wtk1395_err;
        if (test != 0)
            err_code = test;
    }
    *value = err_code;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Test                                                          */
/* Purpose: This function performs either a hardware self-test or a        */
/*          destructive test of trace Memory, and returns a bit-weighted   */
/*          error code in hexadecimal format (refer to Appendix B of the   */
/*          operations manual for the error codes.)                        */
/*=========================================================================*/
int wtk1395_test (int instrID, int function, int *result)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_invalid_integer_range (function, 0, 1, -2) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    if (function !=0)
        Fmt (cmd, "%s<TEST:RAM?");
    else
        Fmt (cmd, "%s<TEST?");

    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    response = 0;
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 50) != 0)
        return wtk1395_err;

    if (Scan (cmd,"%s>%x[b4]", result) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Calibrate                                                     */
/* Purpose: This function performs a self-calibration using default        */
/*          correction values and returns a bit-weighted error code in     */
/*          hexadecimal format (refer to Appendix A of the operations      */
/*          manual for error codes.)                                       */
/*=========================================================================*/
int wtk1395_cal (int instrID, int *result)
{
    char response;

    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<*CAL?");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Serial Polling for Message Available Bit */
    while (response & 0x10 != 0x10) {
        if (wtk1395_poll (instrID, &response) != 0)
            return wtk1395_err;
    }
    if (wtk1395_read_data (instrID, cmd, 50) != 0)
        return wtk1395_err;

    if (Scan (cmd,"%s>%x[b4]", result) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Reset                                                         */
/* Purpose:  This function resets the instrument and returns all parameters*/
/*           to their default values. Reset may also destroy trace data    */
/*           stored in trace memory.                                       */
/*=========================================================================*/
int wtk1395_reset (int instrID)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    Fmt (cmd, "%s<*RST");
    if (wtk1395_write_data(instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

 /* Check Event Status Byte of instrument for proper execution */
    if (wtk1395_check_instr_status (instrID) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Write To Instrument                                           */
/* Purpose:  This function writes a command string to the instrument.      */
/*=========================================================================*/
int wtk1395_write (int instrID, char cmd_string[])
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed(instrID) != 0)
        return wtk1395_err;

    Fmt (cmd, "%s<%s", cmd_string);
    if (wtk1395_write_data (instrID, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Read Instrument Buffer                                        */
/* Purpose:  This function reads the output buffer of the instrument.      */
/*=========================================================================*/
int wtk1395_read (int instrID, int numbytes, char in_buff[], long *bytes_read)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed(instrID) != 0)
        return wtk1395_err;

    *bytes_read = 0L;
    if (wtk1395_read_data (instrID, in_buff, numbytes) != 0)
        return wtk1395_err;

    switch (interface[instrID]) {
        case NI_VXI: *bytes_read = bytes;
                     break;
       case NI_GPIB: *bytes_read = (long)ibcnt;
                     break;
            default: *bytes_read = 0L;
                     break;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int wtk1395_close (int instrID)
{
    if (wtk1395_invalid_integer_range (instrID, 1, wtk1395_MAX_INSTR, -1) != 0)
        return wtk1395_err;
    if (wtk1395_device_closed (instrID))
        return wtk1395_err;

    wtk1395_close_instr (instrID);

    return wtk1395_err;
}

/*====== UTILITY ROUTINES =================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table.  The variable platform equals 1 for NI-VXI  */
/*           and 2 for NI-GPIB. The size of the Instrument Table can be    */
/*           changed in the include file by altering the constant          */
/*           wtk1395_MAX_INSTR. The return value of this function is equal  */
/*           to the global error variable.                                 */
/*=========================================================================*/
int wtk1395_open_instr (int laddr, int platform, int *id)
{
    int i, instrID;
    instrID = wtk1395_err = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= wtk1395_MAX_INSTR; i++)
        if (log_addr[i] == laddr) {
            instrID = i;
            i = wtk1395_MAX_INSTR;
         }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= wtk1395_MAX_INSTR; i++)
            if (log_addr[i] == 0) {
                instrID = i;
                log_addr[instrID] = laddr;
                base_addr[instrID] = (laddr * 0x40) + (long)(0xc000);
                interface[instrID] = platform;
                instr_cnt += 1;
                i = wtk1395_MAX_INSTR;
             }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0) {
        wtk1395_err = 220;
        return wtk1395_err;
     }

/* Initialize the interface. */

    switch (platform) {
        case NI_VXI:  if (InitVXIlibrary() < 0) {
                        wtk1395_close_instr (instrID);
                        wtk1395_err = 222;
                        return wtk1395_err;
                      }
                      break;
       case NI_GPIB:  if (wtk1395_init_gpib (instrID) != 0) {
                        wtk1395_close_instr (instrID);
                        return wtk1395_err;
                      }
                      break;
    }

    *id = instrID;
    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table, if applicable (NI-GPIB only), and setting  */
/*           the logical address, bd, and interface to zero in the         */
/*           Instrument Table. Upon successful completion, zero is         */
/*           returned, otherwise -1 is returned.                           */
/*=========================================================================*/
int wtk1395_close_instr (int instrID)
{
    if (interface[instrID] == NI_GPIB)  {
        if (bd[instrID] > 0)
            if (CloseDev (bd[instrID]) < 0) {
                wtk1395_err = 221;
                return -1;
            }
    }  
    else 
        if (CloseVXIlibrary () < 0) {
           wtk1395_err = 221;
            return  -1;
        }

    log_addr[instrID] = 0;
    bd[instrID] = 0;
    base_addr[instrID] = 0;
    interface[instrID] = 0;
    instr_cnt -= 1;

    return 0;
}

/*=========================================================================*/
/*                         NI-GPIB ONLY                                    */
/* Function: Initialize GPIB                                               */
/* Purpose:  This function initializes the GPIB interface.  First, the     */
/*           GPIB-VXI interface is initialized, then the Instrument        */
/*           interface is initialized.  The primary address of all entries */
/*           is set to 1.  The secondary address the GPIB-VXI is 0 and the */
/*           secondary address of the instrument is obtained by querying   */
/*           the GPIB-VXI.  The return value of the function is equal to   */
/*           the global error variable.                                    */
/*=========================================================================*/
int wtk1395_init_gpib (int instrID)
{
    int sec_addr;
    wtk1395_err = 0;

/* If the GPIB-VXI is not opened (bd[0]=0), open it and initialize its      */
/* primary and secondary addresses.                                         */

    if (bd[0] <= 0)    {
        CloseInstrDevs("wtk1395");
        /* Set the interface of the GPIB-VXI.*/
        interface[0] = NI_GPIB;
        bd[0] = OpenDev ("", "wtk1395");
        if (bd[0] <= 0)   {
            wtk1395_err = 220 ;
            return wtk1395_err;
        }
        if (ibpad (bd[0], 1) & 0x8000)   {
            wtk1395_err = 233;
            return wtk1395_err;
        }

        if (ibsad (bd[0], 0x60) & 0x8000)   {
            wtk1395_err = 233;
            return wtk1395_err;
        }
    }

/* If the instrument is not opened (bd[instrID]=0), open it and initialize  */
/* its primary and secondary addresses.                                     */

    if (bd[instrID] <= 0)    {
        bd[instrID] = OpenDev ("", "wtk1395");
        if (bd[instrID] <= 0)   {
            wtk1395_err = 220 ;
            return wtk1395_err;
        }
    }

/* Query the GPIB-VXI for the secondary address of the instrument. */

    Fmt (cmd, "%s<LaSaddr? %d[b4]", log_addr[instrID]);
    if (wtk1395_write_data (0, cmd, NumFmtdBytes()) != 0)
        return wtk1395_err;

    FillBytes (cmd, 0, 50, 0);

    if (wtk1395_read_data (0, cmd, 50) != 0)
        return wtk1395_err;

    if (Scan (cmd,"%s>%d[b4]", &sec_addr) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
     }

    if (ibpad (bd[instrID], 1) & 0x8000)   {
        wtk1395_err = 233;
        return wtk1395_err;
    }

    if (sec_addr != 255) {
        if (ibsad (bd[instrID], sec_addr + 0x60) & 0x8000)   {
            wtk1395_err = 233;
            return wtk1395_err;
        }
    }
    else
        if (ibsad (bd[instrID], 0x60) & 0x8000)   {
            wtk1395_err = 233;
            return wtk1395_err;
        }

    if (ibconfig (bd[instrID], 6, 1) & 0x8000)   {
        wtk1395_err = 241;
        return wtk1395_err;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int wtk1395_invalid_integer_range (int val, int min, int max, int err_code)
{
    if (val < min || val > max)
        wtk1395_err = err_code;
    else
        wtk1395_err = 0;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Invalid Long Integer Range                                    */
/* Purpose:  This function checks a long integer to see if it lies between */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int wtk1395_invalid_longint_range (long val, long min, long max, int err_code)
{
    if (val < min || val > max)
        wtk1395_err = err_code;
    else
        wtk1395_err = 0;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int wtk1395_invalid_real_range (double val, double min, double max, int err_code)
{
    if ((val < min) || (val > max))
        wtk1395_err = err_code;
    else
        wtk1395_err = 0;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int wtk1395_read_data (int instrID, char *buf, int cnt)
{
    long bytes;
    wtk1395_err = 0;

    switch (interface[instrID]) {

        case NI_VXI: if (WSrd (log_addr[instrID], buf, (long)cnt, 1, &bytes) & 0x8000)
                        wtk1395_err = 231;
                     break;
       case NI_GPIB: if (ibrd(bd[instrID], buf, (long)cnt) & 0x8000)
                        wtk1395_err = 231;
                     break;
            default: wtk1395_err = 231;
                     break;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int wtk1395_write_data (int instrID, char *buf, int cnt)
{
    long bytes;
    wtk1395_err = 0;

    switch (interface[instrID]) {

        case NI_VXI: if ((WSwrt (log_addr[instrID], buf, (long)cnt, 3, &bytes) & 0x8000) || (bytes != cnt))
                        wtk1395_err = 230;
                     break;
       case NI_GPIB: if (ibwrt (bd[instrID], buf, (long)cnt) & 0x8000)
                        wtk1395_err = 230;
                     break;
            default: wtk1395_err = 230;
                     break;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\lw\instr\file" or a pointer   */
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
int wtk1395_read_data_file (int instrID, char *filename)
{
    long bytes;
    wtk1395_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WSrdf (log_addr[instrID], filename, -1L, 1, &bytes) & 0x8000)
                        wtk1395_err = 229;
                     break;
       case NI_GPIB: if (ibrdf (bd[instrID], filename) & 0x8000)
                        wtk1395_err = 229;
                     break;
            default: wtk1395_err = 229;
                     break;
     }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int wtk1395_write_data_file (int instrID, char *filename)
{
    long bytes;
    wtk1395_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WSwrtf (log_addr[instrID], filename, -1L, 1, &bytes) & 0x8000)
                        wtk1395_err = 228;
                     break;
       case NI_GPIB: if (ibwrtf (bd[instrID], filename) & 0x8000)
                        wtk1395_err = 228;
                     break;
            default: wtk1395_err = 228;
                     break;
     }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Read Register                                                 */
/* Purpose:  This function reads an instrument register.  The return value */
/*           is equal to the global error variable.                        */
/*=========================================================================*/
int wtk1395_read_reg (int instrID, int rel_addr, int *value)
{
    unsigned short temp_val = 0;
    wtk1395_err = 0;

    switch (interface[instrID]) {
        case NI_VXI:
            if (VXIinReg (log_addr[instrID], rel_addr, &temp_val) < 0)
                wtk1395_err = 231;
            *value = (int)temp_val;
            break;
        case NI_GPIB:
            Fmt (cmd, "RREG? %d[b4],%d[b4]", log_addr[instrID], rel_addr);
            if (ibwrt(bd[0], cmd, (long)NumFmtdBytes()) & 0x8000)
                wtk1395_err = 230;
            if (ibrd(bd[0], cmd, 20L) & 0x8000)
                wtk1395_err = 231;
            if (Scan (cmd, "%s>%x[b4]", value) != 1)
                wtk1395_err = 236;
            break;
        default:
            wtk1395_err = 231;
            break;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the instrument has been        */
/*           initialized. The return value is equal to the global error    */
/*           variable.                                                     */
/*=========================================================================*/
int wtk1395_device_closed (int instrID)
{
    if (log_addr[instrID] <= 0)
        wtk1395_err = 232;
    else
        wtk1395_err = 0;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Set Timeout                                                   */
/* Purpose:  This function changes or disables the timeout of the device.  */
/*           Refer to array at the end of this file for timeout codes      */
/*           (index of timeout array).                                     */
/*=========================================================================*/
int wtk1395_set_timeout (int instrID, int tmo_code)
{
    long actual_tmo;
    wtk1395_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WSsetTmo (timeout[tmo_code], &actual_tmo) < 0)
                        wtk1395_err = 239;
                     break;
       case NI_GPIB: if (ibtmo (bd[instrID], tmo_code) & 0x8000)
                        wtk1395_err = 239;
                     break;
            default: wtk1395_err = 239;
                     break;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Poll                                                          */
/* Purpose:  This function performs a serial poll on the instrument.  The  */
/*           status byte of the instrument is placed in the response       */
/*           variable. The return value is the global error variable.      */
/*=========================================================================*/
int wtk1395_poll (int instrID, char *response)
{
    unsigned short temp_val = 0;

    wtk1395_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WScmd (log_addr[instrID], 0xcfff, 1, &temp_val) & 0x8000)
                        wtk1395_err = 226;
                     *response = (char) temp_val;
                     break;
       case NI_GPIB: if (ibrsp (bd[instrID], response) & 0x8000)
                        wtk1395_err = 226;
                     break;
            default: wtk1395_err = 226;
                     break;
     }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Get Status Register                                           */
/* Purpose:  This function reads the status register of the instrument.    */
/*           This register is located at base_addr + 4.                    */
/*           The return value is the global error variable.                */
/*=========================================================================*/
int wtk1395_get_status_reg (int instrID, int *response)
{
    wtk1395_err = 0;

    if (wtk1395_read_reg (instrID, 4, response) != 0) {
        wtk1395_err = 226;
        return wtk1395_err;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function:    Write Data w/o EOI                                         */
/* Description: Writes a buffer of data to the instrument. The return      */
/*              value is equal to the global error variable.  This function*/
/*              writes the data buffer but does not assert an end signal   */
/*              or EOI when complete.                                      */
/*=========================================================================*/
int wtk1395_write_data_no_end_sgnl (int instrID, char *buf, int cnt)
{
    long bytes;
    wtk1395_err = 0;

    switch (interface[instrID]) {

        case NI_VXI:
                     if ((WSwrt (log_addr[instrID], buf, (long)cnt, 1, &bytes) & 0x8000) || (bytes != cnt))
                        wtk1395_err = 230;
                     break;
       case NI_GPIB:
                     ibeot (bd[instrID], 0);
                     if (ibwrt (bd[instrID], buf, (long)cnt) & 0x8000)
                        wtk1395_err = 230;
                     ibeot (bd[instrID], 1);
                     break;
            default: wtk1395_err = 230;
                     break;
    }

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Read Error                                                    */
/* Purpose:  This function reads an error code from the instrument error   */
/*           queue.                                                        */
/*=========================================================================*/
int wtk1395_read_err (int instrID, int *err_code)
{

    if (wtk1395_write_data (instrID, ":SYST:ERR?", 10) != 0)
        return wtk1395_err;

    if (wtk1395_read_data (instrID, cmd, 100) != 0)
        return wtk1395_err;
    if (Scan (cmd, "%s>%d[b4]", err_code) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }
    if (*err_code != 0)
        *err_code = (*err_code * -1) + 200;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Check Instrument Status                                       */
/* Purpose:  Checks the status of the instrument and sets wtk1395_err      */
/*           accordingly.                                                  */
/*=========================================================================*/
int wtk1395_check_instr_status (int instrID)
{
    int event_status;
    event_status = wtk1395_err = 0;

 /* Query the instrument for the Event Status Register contents. */
    if (wtk1395_write_data (instrID, "*ESR?", 5) != 0)
        return wtk1395_err;

    if (wtk1395_read_data (instrID, cmd, 100) != 0)
        return wtk1395_err;
    if (Scan (cmd, "%s>%d[b4]", &event_status) != 1) {
        wtk1395_err = 236;
        return wtk1395_err;
    }

    if ((event_status & 0x3C) != 0)
        wtk1395_err = 300;

    return wtk1395_err;
}

/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           static arrays.  This routine should be modified for each      */
/*           instrument to include instrument dependent command arrays.    */
/*=========================================================================*/
void wtk1395_setup_arrays ()
{
    /* Timeout array values are equivalent to timeout in milliseconds. */

    timeout[0] = 0xffffffff;  /* Equivalent to turning timeout off */
    timeout[1] = 1;
    timeout[2] = 1;
    timeout[3] = 1;
    timeout[4] = 1;
    timeout[5] = 1;
    timeout[6] = 3;
    timeout[7] = 10;
    timeout[8] = 30;
    timeout[9] = 100;
    timeout[10] = 300;
    timeout[11] = 1000;
    timeout[12] = 3000;
    timeout[13] = 10000;
    timeout[14] = 30000;
    timeout[15] = 100000;
    timeout[16] = 300000;
    timeout[17] = 1000000;

    adv[0] = "TRIG";
    adv[1] = "AUTO";

    dir[0] = "DOWN";
    dir[1] = "UP";

    sweep[0] = "CRES";
    sweep[1] = "TRES";
    sweep[2] = "HRES";
    sweep[3] = "CREV";
    sweep[4] = "TREV";
    sweep[5] = "HREV";

    space[0] = "LOG";
    space[1] = "LIN";

    func[0] = "DC";
    func[1] = "SQU";
    func[2] = "HFSQ";
    func[3] = "SIN";
    func[4] = "PHS";
    func[5] = "NHS";
    func[6] = "PRAM";
    func[7] = "NRAM";
    func[8] = "PRN";
    func[9] = "TRI";
    func[10] = "WTST";

    source[0] = "INT";
    source[1] = "EXT";
    source[2] = "TTLT0";
    source[3] = "TTLT1";
    source[4] = "TTLT2";
    source[5] = "TTLT3";
    source[6] = "TTLT4";
    source[7] = "TTLT5";
    source[8] = "TTLT6";
    source[9] = "TTLT7";
    source[10] = "CHA";
    source[11] = "ECH";

    adv_source[0] = "INT";
    adv_source[1] = "EXT";
    adv_source[2] = "CHA";
    adv_source[3] = "ECH";

    refosc[0] = "INT";
    refosc[1] = "EXT";
    refosc[2] = "ECLT0";
    refosc[3] = "ECLT1";

    trac_mode[0] = "RAST";
    trac_mode[1] = "CW";

    boolean[0] = "OFF";
    boolean[1] = "ON";

    modulation[0] = "SCM";
    modulation[1] = "AM";

    sync[0] = "ASYN";
    sync[1] = "SYNC";

    qual[0] = "BIT;MARK TRIG";
    qual[1] = "BIT;MARK POS";
    qual[2] = "BCOM";
    qual[3] = "LCOM";
    qual[4] = "INT";

    clk[0] = "FREQ";
    clk[1] = "SOUR RAST";
    clk[2] = "SOUR SYNT";

    sync_marker[0] = "BBIT";
    sync_marker[1] = "ZCR";
}

/*========END==============================================================*/
