 /*
 *  loop.S:
 *
 *	- loop_fkt: DDS loop function
 *  - core1_sio_irq: interupt routine to receive phase increment 
 *    from main program via inter-core FIFO
 *  - get_ddstable: provides DDSTable for main program
 *  
 *  uses R2R DAC on GPIO0 .. GPIO7
 *  based on the work of Martin Ossmann (elektor 03-2012)
 *  
 *  March 2023 Jürgen Hof, DL8TO
 *
 ***********************************************************************
 */
#include "hardware/regs/addressmap.h"
#include "hardware/regs/m0plus.h"
#include "hardware/regs/sio.h"

.section ".scratch_x.", "ax"

.EQU sio_irq_proc1,        0x10
.EQU sio_irq_proc1_offset, 0x80


.thumb_func @ Needed since SDK uses BX to call us
.global get_ddstable
.global loop_fkt 		@ Provide function starting address 

@
@ Assembler function DDS loop_fkt
@ 
@
@ R0 - misc.
@ R1 - Phase Akku
@ R2 - address of sintable
@ R3 - result
@ R4 - delta     
@

loop_fkt:

    PUSH {R4}
	
    @ multicore_fifo_clear_irq()  Macro!
	LDR R3, siobase
	MOV  R1, #255 
    STR  R1, [R3, #SIO_FIFO_ST_OFFSET]	
	
	
	@ Set IRQ Handler to our routine	
	MOV R0, #16
	LDR R1, adr_core1_sio_irq	
	BL irq_set_exclusive_handler
	
	
    @ Enable sio proc1 irq 
	MOV R0, #sio_irq_proc1
	MOV R1, #1	
	BL irq_set_enabled


	@ set default value for delta (1kHz)
	LDR R0, adr_ddsdelta  @ address we want
	LDR R4, [R0] 		  @ set delta                

	MOV R1, #0            @ initialize phase akku
	
	LDR R0, siobase       @ address we want
	Add R0, #16           @ output port address
		
	LDR  R2, adr_ddstable @ load adress of ddstable
	

loop:
					
	ADD  R1, R4           @ Add delta to phase akku   1 cyc.
	LSR  R3, R1, #22 	  @ shift right               1 cyc.
    LDRB R5, [R2, R3] 	  @ Get wave value            2 cyc.
	STRB R5, [R0]	      @ send to port              1 cyc. 	
	B  loop 		   	  @ loop for ever             2 cyc.
						  @ total 7 cycles 
	POP {R4}              @ never reached
	BX LR


.align 4 @ necessary alignment


@
@ Assembler function core1_sio_irq
@ receives new DDSDelta from FIFO
@	
.thumb_func @ necessary for interrupt handlers
@ Core 1 SIO interrupt handler 

core1_sio_irq:
    PUSH {LR}

fifo_pop:
@ If there is data in the fifo, then read it.
	LDR R1, siobase
	LDR R0, [R1, #SIO_FIFO_ST_OFFSET]
	MOV R2, #SIO_FIFO_ST_VLD_BITS
	AND R0, R2
	BNE gotone
	B fifo_pop 			@ try again if woken
	
gotone: LDR R4, [R1, #SIO_FIFO_RD_OFFSET]
	
    @ multicore_fifo_clear_irq()
	LDR R3, siobase
	MOV  R1, #255 
    STR  R1, [R3, #SIO_FIFO_ST_OFFSET]		
	
	POP {PC}             @ return from interrupt


@
@ Assembler function get_ddstable
@ provides pointer to DDDTable  
@	

.align 4 @ necessary alignment
.thumb_func @ Needed since SDK uses BX to call us
get_ddstable:
	LDR  R0, adr_ddstable    @load adress of ddstable	 
	BX   LR                  @ return to caller
	
	
.align 4 
siobase: .word SIO_BASE @ base of the GPIO registers
ppbbase: .word PPB_BASE
disableint: .word M0PLUS_NVIC_ICER_OFFSET
vtoroffset: .word M0PLUS_VTOR_OFFSET
clearint: .word M0PLUS_NVIC_ICPR_OFFSET
setint: .word M0PLUS_NVIC_ISER_OFFSET
	
adr_ddstable: .word loc_ddstable
adr_ddsdelta: .word ddsdelta
adr_core1_sio_irq: .word core1_sio_irq+1
		
.data
.section ".scratch_x.", "ax"

.align 4
ddsdelta:
.word 0x0003AA36

.align 4 				@ necessary alignment
loc_ddstable:

@ 1024 x 8 bit sintable
.byte 0x80,0x80,0x81,0x82,0x83,0x83,0x84,0x85,0x86,0x87,0x87,0x88,0x89,0x8A,0x8A,0x8B
.byte 0x8C,0x8D,0x8E,0x8E,0x8F,0x90,0x91,0x91,0x92,0x93,0x94,0x95,0x95,0x96,0x97,0x98
.byte 0x98,0x99,0x9A,0x9B,0x9B,0x9C,0x9D,0x9E,0x9E,0x9F,0xA0,0xA1,0xA2,0xA2,0xA3,0xA4
.byte 0xA5,0xA5,0xA6,0xA7,0xA7,0xA8,0xA9,0xAA,0xAA,0xAB,0xAC,0xAD,0xAD,0xAE,0xAF,0xB0
.byte 0xB0,0xB1,0xB2,0xB2,0xB3,0xB4,0xB5,0xB5,0xB6,0xB7,0xB7,0xB8,0xB9,0xBA,0xBA,0xBB
.byte 0xBC,0xBC,0xBD,0xBE,0xBE,0xBF,0xC0,0xC0,0xC1,0xC2,0xC2,0xC3,0xC4,0xC4,0xC5,0xC6
.byte 0xC6,0xC7,0xC8,0xC8,0xC9,0xCA,0xCA,0xCB,0xCB,0xCC,0xCD,0xCD,0xCE,0xCF,0xCF,0xD0
.byte 0xD0,0xD1,0xD2,0xD2,0xD3,0xD3,0xD4,0xD5,0xD5,0xD6,0xD6,0xD7,0xD7,0xD8,0xD9,0xD9
.byte 0xDA,0xDA,0xDB,0xDB,0xDC,0xDC,0xDD,0xDD,0xDE,0xDE,0xDF,0xE0,0xE0,0xE1,0xE1,0xE2
.byte 0xE2,0xE3,0xE3,0xE4,0xE4,0xE4,0xE5,0xE5,0xE6,0xE6,0xE7,0xE7,0xE8,0xE8,0xE9,0xE9
.byte 0xEA,0xEA,0xEA,0xEB,0xEB,0xEC,0xEC,0xEC,0xED,0xED,0xEE,0xEE,0xEE,0xEF,0xEF,0xF0
.byte 0xF0,0xF0,0xF1,0xF1,0xF1,0xF2,0xF2,0xF2,0xF3,0xF3,0xF3,0xF4,0xF4,0xF4,0xF5,0xF5
.byte 0xF5,0xF6,0xF6,0xF6,0xF6,0xF7,0xF7,0xF7,0xF8,0xF8,0xF8,0xF8,0xF9,0xF9,0xF9,0xF9
.byte 0xFA,0xFA,0xFA,0xFA,0xFA,0xFB,0xFB,0xFB,0xFB,0xFB,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC
.byte 0xFD,0xFD,0xFD,0xFD,0xFD,0xFD,0xFD,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE
.byte 0xFE,0xFE,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF
.byte 0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE
.byte 0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFE,0xFD,0xFD,0xFD,0xFD,0xFD,0xFD
.byte 0xFD,0xFC,0xFC,0xFC,0xFC,0xFC,0xFC,0xFB,0xFB,0xFB,0xFB,0xFB,0xFA,0xFA,0xFA,0xFA
.byte 0xFA,0xF9,0xF9,0xF9,0xF9,0xF8,0xF8,0xF8,0xF8,0xF7,0xF7,0xF7,0xF6,0xF6,0xF6,0xF6
.byte 0xF5,0xF5,0xF5,0xF4,0xF4,0xF4,0xF3,0xF3,0xF3,0xF2,0xF2,0xF2,0xF1,0xF1,0xF1,0xF0
.byte 0xF0,0xF0,0xEF,0xEF,0xEE,0xEE,0xEE,0xED,0xED,0xEC,0xEC,0xEC,0xEB,0xEB,0xEA,0xEA
.byte 0xEA,0xE9,0xE9,0xE8,0xE8,0xE7,0xE7,0xE6,0xE6,0xE5,0xE5,0xE4,0xE4,0xE4,0xE3,0xE3
.byte 0xE2,0xE2,0xE1,0xE1,0xE0,0xE0,0xDF,0xDE,0xDE,0xDD,0xDD,0xDC,0xDC,0xDB,0xDB,0xDA
.byte 0xDA,0xD9,0xD9,0xD8,0xD7,0xD7,0xD6,0xD6,0xD5,0xD5,0xD4,0xD3,0xD3,0xD2,0xD2,0xD1
.byte 0xD0,0xD0,0xCF,0xCF,0xCE,0xCD,0xCD,0xCC,0xCB,0xCB,0xCA,0xCA,0xC9,0xC8,0xC8,0xC7
.byte 0xC6,0xC6,0xC5,0xC4,0xC4,0xC3,0xC2,0xC2,0xC1,0xC0,0xC0,0xBF,0xBE,0xBE,0xBD,0xBC
.byte 0xBC,0xBB,0xBA,0xBA,0xB9,0xB8,0xB7,0xB7,0xB6,0xB5,0xB5,0xB4,0xB3,0xB2,0xB2,0xB1
.byte 0xB0,0xB0,0xAF,0xAE,0xAD,0xAD,0xAC,0xAB,0xAA,0xAA,0xA9,0xA8,0xA7,0xA7,0xA6,0xA5
.byte 0xA5,0xA4,0xA3,0xA2,0xA2,0xA1,0xA0,0x9F,0x9E,0x9E,0x9D,0x9C,0x9B,0x9B,0x9A,0x99
.byte 0x98,0x98,0x97,0x96,0x95,0x95,0x94,0x93,0x92,0x91,0x91,0x90,0x8F,0x8E,0x8E,0x8D
.byte 0x8C,0x8B,0x8A,0x8A,0x89,0x88,0x87,0x87,0x86,0x85,0x84,0x83,0x83,0x82,0x81,0x80
.byte 0x7F,0x7F,0x7E,0x7D,0x7C,0x7C,0x7B,0x7A,0x79,0x78,0x78,0x77,0x76,0x75,0x75,0x74
.byte 0x73,0x72,0x71,0x71,0x70,0x6F,0x6E,0x6E,0x6D,0x6C,0x6B,0x6A,0x6A,0x69,0x68,0x67
.byte 0x67,0x66,0x65,0x64,0x64,0x63,0x62,0x61,0x61,0x60,0x5F,0x5E,0x5D,0x5D,0x5C,0x5B
.byte 0x5A,0x5A,0x59,0x58,0x58,0x57,0x56,0x55,0x55,0x54,0x53,0x52,0x52,0x51,0x50,0x4F
.byte 0x4F,0x4E,0x4D,0x4D,0x4C,0x4B,0x4A,0x4A,0x49,0x48,0x48,0x47,0x46,0x45,0x45,0x44
.byte 0x43,0x43,0x42,0x41,0x41,0x40,0x3F,0x3F,0x3E,0x3D,0x3D,0x3C,0x3B,0x3B,0x3A,0x39
.byte 0x39,0x38,0x37,0x37,0x36,0x35,0x35,0x34,0x34,0x33,0x32,0x32,0x31,0x30,0x30,0x2F
.byte 0x2F,0x2E,0x2D,0x2D,0x2C,0x2C,0x2B,0x2A,0x2A,0x29,0x29,0x28,0x28,0x27,0x26,0x26
.byte 0x25,0x25,0x24,0x24,0x23,0x23,0x22,0x22,0x21,0x21,0x20,0x1F,0x1F,0x1E,0x1E,0x1D
.byte 0x1D,0x1C,0x1C,0x1B,0x1B,0x1B,0x1A,0x1A,0x19,0x19,0x18,0x18,0x17,0x17,0x16,0x16
.byte 0x15,0x15,0x15,0x14,0x14,0x13,0x13,0x13,0x12,0x12,0x11,0x11,0x11,0x10,0x10,0x0F
.byte 0x0F,0x0F,0x0E,0x0E,0x0E,0x0D,0x0D,0x0D,0x0C,0x0C,0x0C,0x0B,0x0B,0x0B,0x0A,0x0A
.byte 0x0A,0x09,0x09,0x09,0x09,0x08,0x08,0x08,0x07,0x07,0x07,0x07,0x06,0x06,0x06,0x06
.byte 0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x03,0x03
.byte 0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01
.byte 0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
.byte 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
.byte 0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02
.byte 0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05
.byte 0x05,0x06,0x06,0x06,0x06,0x07,0x07,0x07,0x07,0x08,0x08,0x08,0x09,0x09,0x09,0x09
.byte 0x0A,0x0A,0x0A,0x0B,0x0B,0x0B,0x0C,0x0C,0x0C,0x0D,0x0D,0x0D,0x0E,0x0E,0x0E,0x0F
.byte 0x0F,0x0F,0x10,0x10,0x11,0x11,0x11,0x12,0x12,0x13,0x13,0x13,0x14,0x14,0x15,0x15
.byte 0x15,0x16,0x16,0x17,0x17,0x18,0x18,0x19,0x19,0x1A,0x1A,0x1B,0x1B,0x1B,0x1C,0x1C
.byte 0x1D,0x1D,0x1E,0x1E,0x1F,0x1F,0x20,0x21,0x21,0x22,0x22,0x23,0x23,0x24,0x24,0x25
.byte 0x25,0x26,0x26,0x27,0x28,0x28,0x29,0x29,0x2A,0x2A,0x2B,0x2C,0x2C,0x2D,0x2D,0x2E
.byte 0x2F,0x2F,0x30,0x30,0x31,0x32,0x32,0x33,0x34,0x34,0x35,0x35,0x36,0x37,0x37,0x38
.byte 0x39,0x39,0x3A,0x3B,0x3B,0x3C,0x3D,0x3D,0x3E,0x3F,0x3F,0x40,0x41,0x41,0x42,0x43
.byte 0x43,0x44,0x45,0x45,0x46,0x47,0x48,0x48,0x49,0x4A,0x4A,0x4B,0x4C,0x4D,0x4D,0x4E
.byte 0x4F,0x4F,0x50,0x51,0x52,0x52,0x53,0x54,0x55,0x55,0x56,0x57,0x58,0x58,0x59,0x5A
.byte 0x5A,0x5B,0x5C,0x5D,0x5D,0x5E,0x5F,0x60,0x61,0x61,0x62,0x63,0x64,0x64,0x65,0x66
.byte 0x67,0x67,0x68,0x69,0x6A,0x6A,0x6B,0x6C,0x6D,0x6E,0x6E,0x6F,0x70,0x71,0x71,0x72
.byte 0x73,0x74,0x75,0x75,0x76,0x77,0x78,0x78,0x79,0x7A,0x7B,0x7C,0x7C,0x7D,0x7E,0x7F


